/**
 * @license
 * Copyright The Closure Library Authors.
 * SPDX-License-Identifier: Apache-2.0
 */


////////////////////////// NOTE ABOUT EDITING THIS FILE ///////////////////////
//                                                                           //
// Any edits to this file must be applied to vec2d.js by running:            //
//   swap_type.sh vec2f.js > vec2d.js                                        //
//                                                                           //
////////////////////////// NOTE ABOUT EDITING THIS FILE ///////////////////////


/**
 * @fileoverview Provides functions for operating on 2 element float (32bit)
 * vectors.
 *
 * The last parameter will typically be the output object and an object
 * can be both an input and output parameter to all methods except where
 * noted.
 *
 * See the README for notes about the design and structure of the API
 * (especially related to performance).
 */

goog.provide('goog.vec.vec2f');
goog.provide('goog.vec.vec2f.Type');

/** @suppress {extraRequire} */
goog.require('goog.vec');


/** @typedef {!goog.vec.Float32} */ goog.vec.vec2f.Type;


/**
 * Creates a vec2f with all elements initialized to zero.
 *
 * @return {!goog.vec.vec2f.Type} The new vec2f.
 */
goog.vec.vec2f.create = function() {
  'use strict';
  return new Float32Array(2);
};


/**
 * Creates a new vec2f initialized with the value from the given array.
 *
 * @param {!Array<number>} vec The source 2 element array.
 * @return {!goog.vec.vec2f.Type} The new vec2f.
 */
goog.vec.vec2f.createFromArray = function(vec) {
  'use strict';
  const newVec = goog.vec.vec2f.create();
  goog.vec.vec2f.setFromArray(newVec, vec);
  return newVec;
};


/**
 * Creates a new vec2f initialized with the supplied values.
 *
 * @param {number} v0 The value for element at index 0.
 * @param {number} v1 The value for element at index 1.
 * @return {!goog.vec.vec2f.Type} The new vector.
 */
goog.vec.vec2f.createFromValues = function(v0, v1) {
  'use strict';
  const vec = goog.vec.vec2f.create();
  goog.vec.vec2f.setFromValues(vec, v0, v1);
  return vec;
};


/**
 * Creates a clone of the given vec2f.
 *
 * @param {!goog.vec.vec2f.Type} vec The source vec2f.
 * @return {!goog.vec.vec2f.Type} The new cloned vec2f.
 */
goog.vec.vec2f.clone = function(vec) {
  'use strict';
  const newVec = goog.vec.vec2f.create();
  goog.vec.vec2f.setFromVec2f(newVec, vec);
  return newVec;
};


/**
 * Initializes the vector with the given values.
 *
 * @param {!goog.vec.vec2f.Type} vec The vector to receive the values.
 * @param {number} v0 The value for element at index 0.
 * @param {number} v1 The value for element at index 1.
 * @return {!goog.vec.vec2f.Type} Return vec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.setFromValues = function(vec, v0, v1) {
  'use strict';
  vec[0] = v0;
  vec[1] = v1;
  return vec;
};


/**
 * Initializes vec2f vec from vec2f src.
 *
 * @param {!goog.vec.vec2f.Type} vec The destination vector.
 * @param {!goog.vec.vec2f.Type} src The source vector.
 * @return {!goog.vec.vec2f.Type} Return vec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.setFromVec2f = function(vec, src) {
  'use strict';
  vec[0] = src[0];
  vec[1] = src[1];
  return vec;
};


/**
 * Initializes vec2f vec from vec2d src (typed as a Float64Array to
 * avoid circular goog.requires).
 *
 * @param {!goog.vec.vec2f.Type} vec The destination vector.
 * @param {Float64Array} src The source vector.
 * @return {!goog.vec.vec2f.Type} Return vec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.setFromVec2d = function(vec, src) {
  'use strict';
  vec[0] = src[0];
  vec[1] = src[1];
  return vec;
};


/**
 * Initializes vec2f vec from Array src.
 *
 * @param {!goog.vec.vec2f.Type} vec The destination vector.
 * @param {Array<number>} src The source vector.
 * @return {!goog.vec.vec2f.Type} Return vec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.setFromArray = function(vec, src) {
  'use strict';
  vec[0] = src[0];
  vec[1] = src[1];
  return vec;
};


/**
 * Performs a component-wise addition of vec0 and vec1 together storing the
 * result into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first addend.
 * @param {!goog.vec.vec2f.Type} vec1 The second addend.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0 or vec1.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.add = function(vec0, vec1, resultVec) {
  'use strict';
  resultVec[0] = vec0[0] + vec1[0];
  resultVec[1] = vec0[1] + vec1[1];
  return resultVec;
};


/**
 * Performs a component-wise subtraction of vec1 from vec0 storing the
 * result into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The minuend.
 * @param {!goog.vec.vec2f.Type} vec1 The subtrahend.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0 or vec1.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.subtract = function(vec0, vec1, resultVec) {
  'use strict';
  resultVec[0] = vec0[0] - vec1[0];
  resultVec[1] = vec0[1] - vec1[1];
  return resultVec;
};


/**
 * Multiplies each component of vec0 with the matching element of vec0
 * storing the products into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first vector.
 * @param {!goog.vec.vec2f.Type} vec1 The second vector.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.componentMultiply = function(vec0, vec1, resultVec) {
  'use strict';
  resultVec[0] = vec0[0] * vec1[0];
  resultVec[1] = vec0[1] * vec1[1];
  return resultVec;
};


/**
 * Divides each component of vec0 with the matching element of vec0
 * storing the divisor into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first vector.
 * @param {!goog.vec.vec2f.Type} vec1 The second vector.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.componentDivide = function(vec0, vec1, resultVec) {
  'use strict';
  resultVec[0] = vec0[0] / vec1[0];
  resultVec[1] = vec0[1] / vec1[1];
  return resultVec;
};


/**
 * Negates vec0, storing the result into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The vector to negate.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.negate = function(vec0, resultVec) {
  'use strict';
  resultVec[0] = -vec0[0];
  resultVec[1] = -vec0[1];
  return resultVec;
};


/**
 * Takes the absolute value of each component of vec0 storing the result in
 * resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The source vector.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to receive the result.
 *     May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.abs = function(vec0, resultVec) {
  'use strict';
  resultVec[0] = Math.abs(vec0[0]);
  resultVec[1] = Math.abs(vec0[1]);
  return resultVec;
};


/**
 * Multiplies each component of vec0 with scalar storing the product into
 * resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The source vector.
 * @param {number} scalar The value to multiply with each component of vec0.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.scale = function(vec0, scalar, resultVec) {
  'use strict';
  resultVec[0] = vec0[0] * scalar;
  resultVec[1] = vec0[1] * scalar;
  return resultVec;
};


/**
 * Returns the magnitudeSquared of the given vector.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The vector.
 * @return {number} The magnitude of the vector.
 */
goog.vec.vec2f.magnitudeSquared = function(vec0) {
  'use strict';
  const x = vec0[0];
  const y = vec0[1];

  return x * x + y * y;
};


/**
 * Returns the magnitude of the given vector.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The vector.
 * @return {number} The magnitude of the vector.
 */
goog.vec.vec2f.magnitude = function(vec0) {
  'use strict';
  const x = vec0[0];
  const y = vec0[1];

  return Math.sqrt(x * x + y * y);
};


/**
 * Normalizes the given vector storing the result into resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The vector to normalize.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to
 *     receive the result. May be vec0.
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.normalize = function(vec0, resultVec) {
  'use strict';
  const x = vec0[0];
  const y = vec0[1];

  const ilen = 1 / Math.sqrt(x * x + y * y);
  resultVec[0] = x * ilen;
  resultVec[1] = y * ilen;
  return resultVec;
};


/**
 * Returns the scalar product of vectors vec0 and vec1.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first vector.
 * @param {!goog.vec.vec2f.Type} vec1 The second vector.
 * @return {number} The scalar product.
 */
goog.vec.vec2f.dot = function(vec0, vec1) {
  'use strict';
  return vec0[0] * vec1[0] + vec0[1] * vec1[1];
};


/**
 * Returns the squared distance between two points.
 *
 * @param {!goog.vec.vec2f.Type} vec0 First point.
 * @param {!goog.vec.vec2f.Type} vec1 Second point.
 * @return {number} The squared distance between the points.
 */
goog.vec.vec2f.distanceSquared = function(vec0, vec1) {
  'use strict';
  const x = vec0[0] - vec1[0];
  const y = vec0[1] - vec1[1];
  return x * x + y * y;
};


/**
 * Returns the distance between two points.
 *
 * @param {!goog.vec.vec2f.Type} vec0 First point.
 * @param {!goog.vec.vec2f.Type} vec1 Second point.
 * @return {number} The distance between the points.
 */
goog.vec.vec2f.distance = function(vec0, vec1) {
  'use strict';
  return Math.sqrt(goog.vec.vec2f.distanceSquared(vec0, vec1));
};


/**
 * Returns a unit vector pointing from one point to another.
 * If the input points are equal then the result will be all zeros.
 *
 * @param {!goog.vec.vec2f.Type} vec0 Origin point.
 * @param {!goog.vec.vec2f.Type} vec1 Target point.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to receive the
 *     results (may be vec0 or vec1).
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.direction = function(vec0, vec1, resultVec) {
  'use strict';
  const x = vec1[0] - vec0[0];
  const y = vec1[1] - vec0[1];
  let d = Math.sqrt(x * x + y * y);
  if (d) {
    d = 1 / d;
    resultVec[0] = x * d;
    resultVec[1] = y * d;
  } else {
    resultVec[0] = resultVec[1] = 0;
  }
  return resultVec;
};


/**
 * Linearly interpolate from vec0 to vec1 according to f. The value of f should
 * be in the range [0..1] otherwise the results are undefined.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first vector.
 * @param {!goog.vec.vec2f.Type} vec1 The second vector.
 * @param {number} f The interpolation factor.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to receive the
 *     results (may be vec0 or vec1).
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.lerp = function(vec0, vec1, f, resultVec) {
  'use strict';
  const x = vec0[0];
  const y = vec0[1];

  resultVec[0] = (vec1[0] - x) * f + x;
  resultVec[1] = (vec1[1] - y) * f + y;
  return resultVec;
};


/**
 * Compares the components of vec0 with the components of another vector or
 * scalar, storing the larger values in resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The source vector.
 * @param {!goog.vec.vec2f.Type|number} limit The limit vector or scalar.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to receive the
 *     results (may be vec0 or limit).
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.max = function(vec0, limit, resultVec) {
  'use strict';
  if (typeof limit === 'number') {
    resultVec[0] = Math.max(vec0[0], limit);
    resultVec[1] = Math.max(vec0[1], limit);
  } else {
    resultVec[0] = Math.max(vec0[0], limit[0]);
    resultVec[1] = Math.max(vec0[1], limit[1]);
  }
  return resultVec;
};


/**
 * Compares the components of vec0 with the components of another vector or
 * scalar, storing the smaller values in resultVec.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The source vector.
 * @param {!goog.vec.vec2f.Type|number} limit The limit vector or scalar.
 * @param {!goog.vec.vec2f.Type} resultVec The vector to receive the
 *     results (may be vec0 or limit).
 * @return {!goog.vec.vec2f.Type} Return resultVec so that operations can be
 *     chained together.
 */
goog.vec.vec2f.min = function(vec0, limit, resultVec) {
  'use strict';
  if (typeof limit === 'number') {
    resultVec[0] = Math.min(vec0[0], limit);
    resultVec[1] = Math.min(vec0[1], limit);
  } else {
    resultVec[0] = Math.min(vec0[0], limit[0]);
    resultVec[1] = Math.min(vec0[1], limit[1]);
  }
  return resultVec;
};


/**
 * Returns true if the components of vec0 are equal to the components of vec1.
 *
 * @param {!goog.vec.vec2f.Type} vec0 The first vector.
 * @param {!goog.vec.vec2f.Type} vec1 The second vector.
 * @return {boolean} True if the vectors are equal, false otherwise.
 */
goog.vec.vec2f.equals = function(vec0, vec1) {
  'use strict';
  return vec0.length == vec1.length && vec0[0] == vec1[0] && vec0[1] == vec1[1];
};
