/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.controller.api;

import com.dedicatedcode.reitti.dto.LocationPoint;
import com.dedicatedcode.reitti.dto.RawLocationDataResponse;
import com.dedicatedcode.reitti.model.geo.RawLocationPoint;
import com.dedicatedcode.reitti.model.security.MagicLinkAccessLevel;
import com.dedicatedcode.reitti.model.security.TokenUser;
import com.dedicatedcode.reitti.model.security.User;
import com.dedicatedcode.reitti.repository.RawLocationPointJdbcService;
import com.dedicatedcode.reitti.repository.UserJdbcService;
import com.dedicatedcode.reitti.service.LocationPointsSimplificationService;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping(value={"/api/v1"})
public class LocationDataApiController {
    private static final Logger logger = LoggerFactory.getLogger(LocationDataApiController.class);
    private final RawLocationPointJdbcService rawLocationPointJdbcService;
    private final LocationPointsSimplificationService simplificationService;
    private final UserJdbcService userJdbcService;

    @Autowired
    public LocationDataApiController(RawLocationPointJdbcService rawLocationPointJdbcService, LocationPointsSimplificationService simplificationService, UserJdbcService userJdbcService) {
        this.rawLocationPointJdbcService = rawLocationPointJdbcService;
        this.simplificationService = simplificationService;
        this.userJdbcService = userJdbcService;
    }

    @GetMapping(value={"/raw-location-points"})
    public ResponseEntity<?> getRawLocationPointsForCurrentUser(@AuthenticationPrincipal User user, @RequestParam(required=false) String date, @RequestParam(required=false) String startDate, @RequestParam(required=false) String endDate, @RequestParam(required=false, defaultValue="UTC") String timezone, @RequestParam(required=false) Integer zoom, @RequestParam(required=false) Double minLat, @RequestParam(required=false) Double maxLat, @RequestParam(required=false) Double minLng, @RequestParam(required=false) Double maxLng) {
        return this.getRawLocationPoints(user, user.getId(), date, startDate, endDate, timezone, zoom, minLat, maxLat, minLng, maxLng);
    }

    @GetMapping(value={"/raw-location-points/{userId}"})
    public ResponseEntity<?> getRawLocationPoints(@AuthenticationPrincipal User user, @PathVariable Long userId, @RequestParam(required=false) String date, @RequestParam(required=false) String startDate, @RequestParam(required=false) String endDate, @RequestParam(required=false, defaultValue="UTC") String timezone, @RequestParam(required=false) Integer zoom, @RequestParam(required=false) Double minLat, @RequestParam(required=false) Double maxLat, @RequestParam(required=false) Double minLng, @RequestParam(required=false) Double maxLng) {
        try {
            List<Object> result;
            ZoneId userTimezone = ZoneId.of(timezone);
            Instant startOfRange = null;
            Instant endOfRange = null;
            if (startDate != null && endDate != null) {
                try {
                    LocalDateTime startTimestamp = LocalDateTime.parse(startDate);
                    LocalDateTime endTimestamp = LocalDateTime.parse(endDate);
                    startOfRange = startTimestamp.atZone(userTimezone).toInstant();
                    endOfRange = endTimestamp.atZone(userTimezone).toInstant();
                }
                catch (DateTimeParseException startTimestamp) {
                    // empty catch block
                }
                if (startOfRange == null && endOfRange == null) {
                    LocalDate selectedStartDate = LocalDate.parse(startDate);
                    LocalDate selectedEndDate = LocalDate.parse(endDate);
                    startOfRange = selectedStartDate.atStartOfDay(userTimezone).toInstant();
                    endOfRange = selectedEndDate.plusDays(1L).atStartOfDay(userTimezone).toInstant().minusMillis(1L);
                }
            } else if (date != null) {
                LocalDate selectedDate = LocalDate.parse(date);
                startOfRange = selectedDate.atStartOfDay(userTimezone).toInstant();
                endOfRange = selectedDate.plusDays(1L).atStartOfDay(userTimezone).toInstant().minusMillis(1L);
            } else {
                return ResponseEntity.badRequest().body(Map.of("error", "Either 'date' or both 'startDate' and 'endDate' must be provided"));
            }
            boolean includeRawLocationPath = true;
            if (user instanceof TokenUser) {
                if (!Objects.equals(user.getId(), userId)) {
                    throw new IllegalAccessException("User not allowed to fetch raw location points for other users");
                }
                includeRawLocationPath = user.getAuthorities().stream().anyMatch(a -> a.equals((Object)MagicLinkAccessLevel.FULL_ACCESS.asAuthority()) || a.equals((Object)MagicLinkAccessLevel.ONLY_LIVE.asAuthority()) || a.equals((Object)MagicLinkAccessLevel.ONLY_LIVE_WITH_PHOTOS.asAuthority()));
            }
            User userToFetchDataFrom = (User)this.userJdbcService.findById(userId).orElseThrow(() -> new RuntimeException("User not found"));
            long start = System.nanoTime();
            if (includeRawLocationPath) {
                List segments = this.loadSegmentsInBoundingBoxAndTime(userToFetchDataFrom, minLat, maxLat, minLng, maxLng, startOfRange, endOfRange);
                logger.trace("Loaded segments in [{}]ms", (Object)((System.nanoTime() - start) / 1000000L));
                start = System.nanoTime();
                result = segments.stream().map(s -> {
                    List simplifiedPoints = this.simplificationService.simplifyPoints(s, zoom);
                    return new RawLocationDataResponse.Segment(s);
                }).toList();
                logger.trace("Simplified segments in [{}]ms", (Object)((System.nanoTime() - start) / 1000000L));
            } else {
                result = Collections.emptyList();
            }
            Optional latest = this.rawLocationPointJdbcService.findLatest(userToFetchDataFrom);
            return ResponseEntity.ok((Object)new RawLocationDataResponse(result, (LocationPoint)latest.map(arg_0 -> this.toLocationPoint(arg_0)).orElse(null)));
        }
        catch (DateTimeParseException e) {
            return ResponseEntity.badRequest().body(Map.of("error", "Invalid date format. Expected format: YYYY-MM-DD"));
        }
        catch (Exception e) {
            logger.error("Error fetching raw location points", (Throwable)e);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.INTERNAL_SERVER_ERROR).body(Map.of("error", "Error fetching raw location points: " + e.getMessage()));
        }
    }

    @GetMapping(value={"/latest-location"})
    public ResponseEntity<?> getLatestLocationForCurrentUser(@AuthenticationPrincipal User user, @RequestParam(required=false) Instant since) {
        try {
            Optional latest = since == null ? this.rawLocationPointJdbcService.findLatest(user) : this.rawLocationPointJdbcService.findLatest(user, since);
            if (latest.isEmpty()) {
                return ResponseEntity.ok(Map.of("hasLocation", false));
            }
            RawLocationPoint latestPoint = (RawLocationPoint)latest.get();
            LocationPoint point = new LocationPoint();
            point.setLatitude(latestPoint.getLatitude());
            point.setLongitude(latestPoint.getLongitude());
            point.setAccuracyMeters(latestPoint.getAccuracyMeters());
            point.setTimestamp(latestPoint.getTimestamp().toString());
            return ResponseEntity.ok(Map.of("hasLocation", true, "point", point));
        }
        catch (Exception e) {
            logger.error("Error fetching latest location", (Throwable)e);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.INTERNAL_SERVER_ERROR).body(Map.of("error", "Error fetching latest location: " + e.getMessage()));
        }
    }

    private List<List<LocationPoint>> loadSegmentsInBoundingBoxAndTime(User user, Double minLat, Double maxLat, Double minLng, Double maxLng, Instant startOfRange, Instant endOfRange) {
        List pointsInBoxWithNeighbors;
        long start = System.nanoTime();
        if (Duration.between(startOfRange, endOfRange).toDays() > 30L && minLat == null && maxLat == null && minLng == null && maxLng == null) {
            pointsInBoxWithNeighbors = this.rawLocationPointJdbcService.findSimplifiedRouteForPeriod(user, startOfRange, endOfRange, 10000);
        } else if (minLat == null || maxLat == null || minLng == null || maxLng == null) {
            pointsInBoxWithNeighbors = this.rawLocationPointJdbcService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, startOfRange, endOfRange);
            logger.trace("Loaded points in time range from database in [{}]ms", (Object)((System.nanoTime() - start) / 1000000L));
        } else {
            pointsInBoxWithNeighbors = this.rawLocationPointJdbcService.findPointsInBoxWithNeighbors(user, startOfRange, endOfRange, minLat.doubleValue(), maxLat.doubleValue(), minLng.doubleValue(), maxLng.doubleValue(), 10000);
            logger.trace("Loaded points in box with neighbors from database in [{}]ms", (Object)((System.nanoTime() - start) / 1000000L));
        }
        return this.extractPathSegments(pointsInBoxWithNeighbors, minLat, maxLat, minLng, maxLng);
    }

    private List<List<LocationPoint>> extractPathSegments(List<RawLocationPoint> points, Double minLat, Double maxLat, Double minLng, Double maxLng) {
        long start = System.nanoTime();
        ArrayList<List<LocationPoint>> segments = new ArrayList<List<LocationPoint>>();
        ArrayList<LocationPoint> currentPath = new ArrayList<LocationPoint>();
        int consecutiveOutside = 0;
        for (RawLocationPoint point : points) {
            boolean inBox = this.isPointInBox(point, minLat, maxLat, minLng, maxLng);
            if (inBox) {
                currentPath.add(this.toLocationPoint(point));
                consecutiveOutside = 0;
                continue;
            }
            currentPath.add(this.toLocationPoint(point));
            if (++consecutiveOutside != 2) continue;
            currentPath.removeLast();
            if (currentPath.size() >= 2) {
                segments.add(new ArrayList(currentPath));
            }
            currentPath.clear();
            consecutiveOutside = 0;
        }
        if (currentPath.size() >= 2) {
            segments.add(new ArrayList(currentPath));
        }
        logger.trace("Extracted path segments in [{}]ms", (Object)((System.nanoTime() - start) / 1000000L));
        return segments;
    }

    private boolean isPointInBox(RawLocationPoint point, Double minLat, Double maxLat, Double minLng, Double maxLng) {
        if (minLat == null || maxLat == null || minLng == null || maxLng == null) {
            return true;
        }
        return point.getLatitude() >= minLat && point.getLatitude() <= maxLat && point.getLongitude() >= minLng && point.getLongitude() <= maxLng;
    }

    private LocationPoint toLocationPoint(RawLocationPoint point) {
        LocationPoint p = new LocationPoint();
        p.setLatitude(point.getLatitude());
        p.setLongitude(point.getLongitude());
        p.setAccuracyMeters(point.getAccuracyMeters());
        p.setElevationMeters(point.getElevationMeters());
        p.setTimestamp(point.getTimestamp().toString());
        return p;
    }
}

