/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service;

import com.dedicatedcode.reitti.service.DefaultImportProcessor;
import com.dedicatedcode.reitti.service.QueueStats;
import com.dedicatedcode.reitti.service.QueueStatsService;
import com.dedicatedcode.reitti.service.processing.ProcessingPipelineTrigger;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

@Service
public class QueueStatsService {
    public static final String STAY_DETECTION_QUEUE = "reitti.visit.detection.v2";
    public static final String LOCATION_DATA_QUEUE = "reitti.location.data.v2";
    private final RabbitAdmin rabbitAdmin;
    private final MessageSource messageSource;
    private final ProcessingPipelineTrigger processingPipelineTrigger;
    private final DefaultImportProcessor defaultImportProcessor;
    private static final int LOOKBACK_HOURS = 24;
    private static final long DEFAULT_PROCESSING_TIME = 2000L;
    private final List<String> QUEUES = List.of("reitti.place.created.v2", "reitti.user.events.v2");
    private final Map<String, List<ProcessingRecord>> processingHistory = new ConcurrentHashMap();
    private final Map<String, Integer> previousMessageCounts = new ConcurrentHashMap();

    public QueueStatsService(RabbitAdmin rabbitAdmin, MessageSource messageSource, ProcessingPipelineTrigger processingPipelineTrigger, DefaultImportProcessor defaultImportProcessor) {
        this.rabbitAdmin = rabbitAdmin;
        this.messageSource = messageSource;
        this.processingPipelineTrigger = processingPipelineTrigger;
        this.defaultImportProcessor = defaultImportProcessor;
        this.QUEUES.forEach(queue -> {
            this.processingHistory.put(queue, new ArrayList());
            this.previousMessageCounts.put(queue, 0);
        });
        this.processingHistory.put(STAY_DETECTION_QUEUE, new ArrayList());
        this.previousMessageCounts.put(STAY_DETECTION_QUEUE, 0);
        this.processingHistory.put(LOCATION_DATA_QUEUE, new ArrayList());
        this.previousMessageCounts.put(LOCATION_DATA_QUEUE, 0);
    }

    public List<QueueStats> getQueueStats() {
        List<QueueStats> list = this.QUEUES.stream().map(arg_0 -> this.getQueueStats(arg_0)).toList();
        ArrayList<QueueStats> result = new ArrayList<QueueStats>(list);
        result.add(0, this.getQueueStats(LOCATION_DATA_QUEUE));
        result.add(1, this.getQueueStats(STAY_DETECTION_QUEUE));
        return result;
    }

    private QueueStats getQueueStats(String name) {
        int currentMessageCount;
        if (name.equals(STAY_DETECTION_QUEUE)) {
            currentMessageCount = this.processingPipelineTrigger.getPendingCount();
            this.updatingStayDetectionQueue(currentMessageCount);
        } else if (name.equals(LOCATION_DATA_QUEUE)) {
            currentMessageCount = this.defaultImportProcessor.getPendingTaskCount();
            this.updatingLocationDataQueue(currentMessageCount);
        } else {
            currentMessageCount = this.getMessageCount(name);
            this.updateProcessingHistoryFromRabbitMQ(name, currentMessageCount);
        }
        long avgProcessingTime = this.calculateAverageProcessingTime(name);
        long estimatedTime = (long)currentMessageCount * avgProcessingTime;
        String displayName = this.getLocalizedDisplayName(name);
        String description = this.getLocalizedDescription(name);
        return new QueueStats(name, displayName, description, currentMessageCount, this.formatProcessingTime(estimatedTime), this.calculateProgress(name, currentMessageCount));
    }

    private void updateProcessingHistoryFromRabbitMQ(String queueName, int currentMessageCount) {
        Integer previousCount = (Integer)this.previousMessageCounts.get(queueName);
        if (previousCount != null && currentMessageCount < previousCount) {
            long processingTimePerMessage = this.estimateProcessingTimePerMessage(queueName);
            List history = (List)this.processingHistory.get(queueName);
            LocalDateTime now = LocalDateTime.now();
            history.add(new ProcessingRecord(now, (long)this.rabbitAdmin.getQueueInfo(queueName).getMessageCount(), processingTimePerMessage));
            this.cleanupOldRecords(history, now);
        }
        this.previousMessageCounts.put(queueName, currentMessageCount);
    }

    private void updatingStayDetectionQueue(int currentMessageCount) {
        Integer previousCount = (Integer)this.previousMessageCounts.get(STAY_DETECTION_QUEUE);
        if (previousCount != null && currentMessageCount < previousCount) {
            long processingTimePerMessage = this.estimateProcessingTimePerMessage(STAY_DETECTION_QUEUE);
            List history = (List)this.processingHistory.get(STAY_DETECTION_QUEUE);
            LocalDateTime now = LocalDateTime.now();
            history.add(new ProcessingRecord(now, (long)this.processingPipelineTrigger.getPendingCount(), processingTimePerMessage));
            this.cleanupOldRecords(history, now);
        }
        this.previousMessageCounts.put(STAY_DETECTION_QUEUE, currentMessageCount);
    }

    private void updatingLocationDataQueue(int currentMessageCount) {
        Integer previousCount = (Integer)this.previousMessageCounts.get(LOCATION_DATA_QUEUE);
        if (previousCount != null && currentMessageCount < previousCount) {
            long processingTimePerMessage = this.estimateProcessingTimePerMessage(LOCATION_DATA_QUEUE);
            List history = (List)this.processingHistory.get(LOCATION_DATA_QUEUE);
            LocalDateTime now = LocalDateTime.now();
            history.add(new ProcessingRecord(now, (long)this.defaultImportProcessor.getPendingTaskCount(), processingTimePerMessage));
            this.cleanupOldRecords(history, now);
        }
        this.previousMessageCounts.put(LOCATION_DATA_QUEUE, currentMessageCount);
    }

    private long estimateProcessingTimePerMessage(String queueName) {
        List history = (List)this.processingHistory.get(queueName);
        if (history.isEmpty()) {
            return 2000L;
        }
        return this.calculateAverageFromHistory(history);
    }

    private long calculateAverageProcessingTime(String queueName) {
        List history = (List)this.processingHistory.get(queueName);
        if (history.isEmpty()) {
            return 2000L;
        }
        return this.calculateAverageFromHistory(history);
    }

    private long calculateAverageFromHistory(List<ProcessingRecord> history) {
        if (history.isEmpty()) {
            return 2000L;
        }
        return (long)history.stream().mapToLong(record -> record.processingTimeMs).average().orElse(2000.0);
    }

    private void cleanupOldRecords(List<ProcessingRecord> history, LocalDateTime now) {
        LocalDateTime cutoff = now.minusHours(24L);
        history.removeIf(record -> record.timestamp.isBefore(cutoff));
    }

    private int getMessageCount(String queueName) {
        Properties properties = this.rabbitAdmin.getQueueProperties(queueName);
        if (properties.containsKey(RabbitAdmin.QUEUE_MESSAGE_COUNT)) {
            return (Integer)properties.get(RabbitAdmin.QUEUE_MESSAGE_COUNT);
        }
        return 0;
    }

    private String formatProcessingTime(long milliseconds) {
        if (milliseconds < 60000L) {
            return milliseconds / 1000L + " sec";
        }
        if (milliseconds < 3600000L) {
            return milliseconds / 60000L + " min";
        }
        long hours = milliseconds / 3600000L;
        long minutes = milliseconds % 3600000L / 60000L;
        return hours + " hr " + minutes + " min";
    }

    private int calculateProgress(String queueName, int currentMessageCount) {
        if (currentMessageCount == 0) {
            return 100;
        }
        List history = (List)this.processingHistory.get(queueName);
        if (history.isEmpty()) {
            if (currentMessageCount <= 5) {
                return 80;
            }
            if (currentMessageCount <= 20) {
                return 60;
            }
            if (currentMessageCount <= 100) {
                return 40;
            }
            return 20;
        }
        Integer previousCount = (Integer)this.previousMessageCounts.get(queueName);
        if (previousCount != null && previousCount > currentMessageCount) {
            int processedRecently = previousCount - currentMessageCount;
            double processingRate = Math.min(100.0, (double)processedRecently / (double)Math.max(1, previousCount) * 100.0);
            return Math.max(50, (int)(50.0 + processingRate / 2.0));
        }
        if (currentMessageCount <= 10) {
            return 70;
        }
        if (currentMessageCount <= 50) {
            return 50;
        }
        if (currentMessageCount <= 200) {
            return 30;
        }
        return 10;
    }

    private String getLocalizedDisplayName(String queueName) {
        String key = this.getMessageKeyForQueue(queueName, "name");
        return this.messageSource.getMessage(key, null, queueName, LocaleContextHolder.getLocale());
    }

    private String getLocalizedDescription(String queueName) {
        String key = this.getMessageKeyForQueue(queueName, "description");
        return this.messageSource.getMessage(key, null, "Processing " + queueName, LocaleContextHolder.getLocale());
    }

    private String getMessageKeyForQueue(String queueName, String suffix) {
        return switch (queueName) {
            case "reitti.place.created.v2" -> "queue.significant.place." + suffix;
            case "reitti.user.events.v2" -> "queue.user.event." + suffix;
            case STAY_DETECTION_QUEUE -> "queue.stay.detection." + suffix;
            case LOCATION_DATA_QUEUE -> "queue.location.data." + suffix;
            default -> "queue.unknown." + suffix;
        };
    }
}

