/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service.geocoding;

import com.dedicatedcode.reitti.model.geo.SignificantPlace;
import com.dedicatedcode.reitti.model.geocoding.GeocodingResponse;
import com.dedicatedcode.reitti.model.geocoding.RemoteGeocodeService;
import com.dedicatedcode.reitti.repository.GeocodeServiceJdbcService;
import com.dedicatedcode.reitti.repository.GeocodingResponseJdbcService;
import com.dedicatedcode.reitti.service.geocoding.GeocodeResult;
import com.dedicatedcode.reitti.service.geocoding.GeocodeService;
import com.dedicatedcode.reitti.service.geocoding.GeocodeServiceManager;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

@Service
public class DefaultGeocodeServiceManager
implements GeocodeServiceManager {
    private static final Logger logger = LoggerFactory.getLogger(DefaultGeocodeServiceManager.class);
    private final GeocodeServiceJdbcService geocodeServiceJdbcService;
    private final GeocodingResponseJdbcService geocodingResponseJdbcService;
    private final List<GeocodeService> fixedGeocodeServices;
    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;
    private final int maxErrors;

    public DefaultGeocodeServiceManager(GeocodeServiceJdbcService geocodeServiceJdbcService, GeocodingResponseJdbcService geocodingResponseJdbcService, List<GeocodeService> fixedGeocodeServices, RestTemplate restTemplate, ObjectMapper objectMapper, @Value(value="${reitti.geocoding.max-errors}") int maxErrors) {
        this.geocodeServiceJdbcService = geocodeServiceJdbcService;
        this.geocodingResponseJdbcService = geocodingResponseJdbcService;
        this.fixedGeocodeServices = fixedGeocodeServices;
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper;
        this.maxErrors = maxErrors;
    }

    @Transactional
    public Optional<GeocodeResult> reverseGeocode(SignificantPlace significantPlace, boolean recordResponse) {
        List availableServices;
        double latitude = significantPlace.getLatitudeCentroid();
        double longitude = significantPlace.getLongitudeCentroid();
        if (!this.fixedGeocodeServices.isEmpty()) {
            logger.debug("Fixed geocode-service available, will first try this.");
            Optional geocodeResult = this.callGeocodeService(this.fixedGeocodeServices, latitude, longitude, true, significantPlace, recordResponse);
            if (geocodeResult.isPresent()) {
                return geocodeResult;
            }
        }
        if ((availableServices = this.geocodeServiceJdbcService.findByEnabledTrueOrderByLastUsedAsc()).isEmpty()) {
            logger.warn("No enabled geocoding services available");
            return Optional.empty();
        }
        return this.callGeocodeService(availableServices, latitude, longitude, false, significantPlace, recordResponse);
    }

    private Optional<GeocodeResult> callGeocodeService(List<? extends GeocodeService> availableServices, double latitude, double longitude, boolean photon, SignificantPlace significantPlace, boolean recordResponse) {
        Collections.shuffle(availableServices);
        for (GeocodeService geocodeService : availableServices) {
            try {
                Optional result = this.performGeocode(geocodeService, latitude, longitude, photon, significantPlace, recordResponse);
                if (!result.isPresent()) continue;
                if (recordResponse) {
                    this.recordSuccess(geocodeService);
                }
                return result;
            }
            catch (Exception e) {
                logger.warn("Geocoding failed for service [{}]: [{}]", (Object)geocodeService.getName(), (Object)e.getMessage());
                if (!recordResponse) continue;
                this.recordError(geocodeService);
            }
        }
        return Optional.empty();
    }

    private Optional<GeocodeResult> performGeocode(GeocodeService service, double latitude, double longitude, boolean photon, SignificantPlace significantPlace, boolean recordResponse) {
        String url = service.getUrlTemplate().replace("{lat}", String.valueOf(latitude)).replace("{lng}", String.valueOf(longitude));
        logger.info("Geocoding with service [{}] using URL: [{}]", (Object)service.getName(), (Object)url);
        try {
            Optional geocodeResult;
            String response = (String)this.restTemplate.getForObject(url, String.class, new Object[0]);
            Optional optional = geocodeResult = photon ? this.extractPhotonResult(response) : this.extractGeoCodeResult(response);
            if (recordResponse && geocodeResult.isPresent()) {
                this.geocodingResponseJdbcService.insert(new GeocodingResponse(significantPlace.getId(), response, service.getName(), Instant.now(), GeocodingResponse.GeocodingStatus.SUCCESS, null));
            } else if (recordResponse) {
                this.geocodingResponseJdbcService.insert(new GeocodingResponse(significantPlace.getId(), response, service.getName(), Instant.now(), GeocodingResponse.GeocodingStatus.ZERO_RESULTS, null));
            }
            return geocodeResult;
        }
        catch (Exception e) {
            logger.error("Failed to call geocoding service [{}]: [{}]", (Object)service.getName(), (Object)e.getMessage());
            GeocodingResponse.GeocodingStatus status = this.determineErrorStatus(e);
            if (recordResponse) {
                this.geocodingResponseJdbcService.insert(new GeocodingResponse(significantPlace.getId(), null, service.getName(), Instant.now(), status, e.getMessage()));
            }
            throw new RuntimeException("Failed to call geocoding service: " + e.getMessage(), e);
        }
    }

    private Optional<GeocodeResult> extractPhotonResult(String response) throws JsonProcessingException {
        JsonNode root = this.objectMapper.readTree(response);
        JsonNode features = root.path("features");
        if (features.isArray() && !features.isEmpty()) {
            JsonNode properties = features.get(0).path("properties");
            String name = properties.path("name").asText();
            String city = properties.path("city").asText();
            String street = properties.path("street").asText();
            String district = properties.path("district").asText();
            String housenumber = properties.path("housenumber").asText();
            String postcode = properties.path("postcode").asText();
            String countryCode = properties.path("countrycode").asText().toLowerCase();
            SignificantPlace.PlaceType type = this.determinPlaceType(properties.path("osm_value").asText());
            return Optional.of(new GeocodeResult(name, street, housenumber, city, postcode, district, countryCode, type));
        }
        return Optional.empty();
    }

    private SignificantPlace.PlaceType determinPlaceType(String osmValue) {
        return switch (osmValue) {
            case "office", "commercial", "industrial", "warehouse", "retail" -> SignificantPlace.PlaceType.WORK;
            case "restaurant", "fast_food", "food_court" -> SignificantPlace.PlaceType.RESTAURANT;
            case "cafe", "bar", "pub" -> SignificantPlace.PlaceType.CAFE;
            case "shop", "supermarket", "mall", "marketplace", "department_store", "convenience" -> SignificantPlace.PlaceType.SHOP;
            case "hospital", "clinic", "doctors", "dentist", "veterinary" -> SignificantPlace.PlaceType.HOSPITAL;
            case "pharmacy" -> SignificantPlace.PlaceType.PHARMACY;
            case "school", "university", "college", "kindergarten" -> SignificantPlace.PlaceType.SCHOOL;
            case "library" -> SignificantPlace.PlaceType.LIBRARY;
            case "gym", "fitness_centre", "sports_centre", "swimming_pool", "stadium" -> SignificantPlace.PlaceType.GYM;
            case "cinema", "theatre" -> SignificantPlace.PlaceType.CINEMA;
            case "park", "garden", "nature_reserve", "beach", "playground" -> SignificantPlace.PlaceType.PARK;
            case "fuel", "charging_station" -> SignificantPlace.PlaceType.GAS_STATION;
            case "bank", "atm", "bureau_de_change" -> SignificantPlace.PlaceType.BANK;
            case "place_of_worship", "church", "mosque", "synagogue", "temple" -> SignificantPlace.PlaceType.CHURCH;
            case "bus_stop", "bus_station", "railway_station", "subway_entrance", "tram_stop" -> SignificantPlace.PlaceType.TRAIN_STATION;
            case "airport", "terminal" -> SignificantPlace.PlaceType.AIRPORT;
            case "hotel", "motel", "guest_house" -> SignificantPlace.PlaceType.HOTEL;
            default -> SignificantPlace.PlaceType.OTHER;
        };
    }

    private Optional<GeocodeResult> extractGeoCodeResult(String response) throws JsonProcessingException {
        JsonNode root = this.objectMapper.readTree(response);
        JsonNode features = root.path("features");
        if (features.isArray() && !features.isEmpty()) {
            Optional result;
            String osmTypeValue;
            String countryCode;
            String district;
            String city;
            Object street;
            String label;
            JsonNode properties = features.get(0).path("properties");
            JsonNode address = properties.path("address");
            JsonNode geocoding = properties.path("geocoding");
            if (geocoding.isObject()) {
                label = geocoding.path("name").asText();
                if (label.isBlank()) {
                    label = geocoding.path("label").asText();
                }
                if (((String)(street = geocoding.path("street").asText())).isBlank()) {
                    street = geocoding.path("road").asText();
                }
                if (geocoding.has("housenumber")) {
                    street = (String)street + " " + geocoding.path("housenumber").asText();
                }
                city = geocoding.path("city").asText();
                district = geocoding.path("city_district").asText();
                if (district.isBlank()) {
                    district = geocoding.path("district").asText();
                }
                if (district.isBlank()) {
                    district = geocoding.path("locality").asText();
                }
                countryCode = geocoding.path("country_code").asText().toLowerCase();
                osmTypeValue = geocoding.path("osm_value").asText();
                if (osmTypeValue.isBlank()) {
                    osmTypeValue = geocoding.path("category").asText();
                }
            } else if (address.isMissingNode()) {
                label = properties.path("formatted").asText("");
                street = properties.path("street").asText("");
                city = properties.path("city").asText("");
                district = properties.path("city_district").asText("");
                countryCode = properties.path("country_code").asText("").toLowerCase();
                osmTypeValue = geocoding.path("osm_value").asText();
                if (osmTypeValue.isBlank()) {
                    osmTypeValue = geocoding.path("category").asText();
                }
            } else {
                label = properties.path("name").asText("");
                street = address.path("road").asText("");
                city = address.path("city").asText("");
                district = address.path("city_district").asText("");
                countryCode = address.path("country_code").asText("").toLowerCase();
                osmTypeValue = geocoding.path("osm_value").asText();
                if (osmTypeValue.isBlank()) {
                    osmTypeValue = geocoding.path("category").asText();
                }
            }
            if ((result = this.createGeoCodeResult(label, (String)street, city, district, countryCode, osmTypeValue)).isPresent()) {
                return result;
            }
        }
        if (root.has("name") && root.has("address")) {
            String countryCode;
            Optional result;
            String label = root.get("name").asText();
            Object street = root.path("address").path("street").asText();
            if (((String)street).isBlank()) {
                street = root.path("address").path("road").asText();
            }
            if (root.path("address").path("house_number").isTextual()) {
                street = (String)street + " " + root.path("address").path("house_number").asText();
            }
            String city = root.path("address").path("city").asText();
            String district = root.path("address").path("district").asText();
            if (district.isBlank()) {
                district = root.path("address").path("neighbourhood").asText();
            }
            if ((result = this.createGeoCodeResult(label, (String)street, city, district, countryCode = root.path("address").path("country_code").asText(), root.path("osm_value").asText())).isPresent()) {
                return result;
            }
        }
        return Optional.empty();
    }

    private Optional<GeocodeResult> createGeoCodeResult(String label, String street, String city, String district, String countryCode, String placeTypeValue) {
        if (label.isEmpty() && !street.isEmpty()) {
            label = street;
        }
        if (StringUtils.hasText((String)label)) {
            return Optional.of(new GeocodeResult(label, street, "", city, "", district, countryCode, this.determinPlaceType(placeTypeValue)));
        }
        return Optional.empty();
    }

    private void recordSuccess(GeocodeService service) {
        if (service instanceof RemoteGeocodeService) {
            this.geocodeServiceJdbcService.save(((RemoteGeocodeService)service).withLastUsed(Instant.now()));
        }
    }

    private GeocodingResponse.GeocodingStatus determineErrorStatus(Exception e) {
        String message = e.getMessage().toLowerCase();
        if (message.contains("rate limit") || message.contains("too many requests") || message.contains("429") || message.contains("quota exceeded")) {
            return GeocodingResponse.GeocodingStatus.RATE_LIMITED;
        }
        if (message.contains("invalid") || message.contains("bad request") || message.contains("400") || message.contains("malformed")) {
            return GeocodingResponse.GeocodingStatus.INVALID_REQUEST;
        }
        return GeocodingResponse.GeocodingStatus.ERROR;
    }

    private void recordError(GeocodeService service) {
        if (service instanceof RemoteGeocodeService) {
            RemoteGeocodeService update = ((RemoteGeocodeService)service).withIncrementedErrorCount().withLastError(Instant.now());
            if (update.getErrorCount() >= this.maxErrors) {
                update = update.withEnabled(false);
                logger.warn("Geocoding service [{}] disabled due to too many errors ([{}]/[{}])", new Object[]{update.getName(), update.getErrorCount(), this.maxErrors});
            }
            this.geocodeServiceJdbcService.save(update);
        }
    }
}

