/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service.processing;

import com.dedicatedcode.reitti.dto.LocationPoint;
import com.dedicatedcode.reitti.model.geo.GeoUtils;
import com.dedicatedcode.reitti.service.processing.GeoPointAnomalyFilterConfig;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class GeoPointAnomalyFilter {
    private static final Logger logger = LoggerFactory.getLogger(GeoPointAnomalyFilter.class);
    private final GeoPointAnomalyFilterConfig config;

    public GeoPointAnomalyFilter(GeoPointAnomalyFilterConfig config) {
        this.config = config;
    }

    public List<LocationPoint> filterAnomalies(List<LocationPoint> points) {
        if (points == null || points.isEmpty()) {
            return new ArrayList<LocationPoint>();
        }
        HashSet detectedAnomalies = new HashSet();
        detectedAnomalies.addAll(this.detectAccuracyAnomalies(points));
        detectedAnomalies.addAll(this.detectSpeedAnomalies(points));
        detectedAnomalies.addAll(this.detectDistanceJumpAnomalies(points));
        detectedAnomalies.addAll(this.detectDirectionAnomalies(points));
        return points.stream().filter(point -> !detectedAnomalies.contains(point)).collect(Collectors.toList());
    }

    private Set<LocationPoint> detectAccuracyAnomalies(List<LocationPoint> points) {
        HashSet<LocationPoint> anomalies = new HashSet<LocationPoint>();
        for (LocationPoint point : points) {
            if (!(point.getAccuracyMeters() > this.config.maxAccuracyMeters)) continue;
            anomalies.add(point);
        }
        logger.debug("Filtering out [{}] points because min accuracy [{}] not met.", (Object)anomalies.size(), (Object)this.config.maxAccuracyMeters);
        return anomalies;
    }

    private Set<LocationPoint> detectSpeedAnomalies(List<LocationPoint> points) {
        HashSet<LocationPoint> anomalies = new HashSet<LocationPoint>();
        for (int i = 1; i < points.size(); ++i) {
            double maxSpeed;
            LocationPoint prev = points.get(i - 1);
            LocationPoint curr = points.get(i);
            if (prev.getTimestamp() == null || curr.getTimestamp() == null) continue;
            double distance = GeoUtils.distanceInMeters((LocationPoint)prev, (LocationPoint)curr);
            long timeDiffSeconds = ChronoUnit.SECONDS.between(this.getTimestamp(prev.getTimestamp()), this.getTimestamp(curr.getTimestamp()));
            if (timeDiffSeconds <= 0L) continue;
            double speedKmh = distance / 1000.0 / ((double)timeDiffSeconds / 3600.0);
            if (this.isEdgePoint(i, points.size())) {
                double d = this.config.maxSpeedKmh;
                Objects.requireNonNull(this.config);
                v1 = d * 1.5;
            } else {
                v1 = maxSpeed = this.config.maxSpeedKmh;
            }
            if (!(speedKmh > maxSpeed)) continue;
            if (curr.getAccuracyMeters() > prev.getAccuracyMeters()) {
                anomalies.add(curr);
                continue;
            }
            anomalies.add(prev);
        }
        logger.debug("Filtering out [{}] points because speed was above [{}].", (Object)anomalies.size(), (Object)this.config.maxSpeedKmh);
        return anomalies;
    }

    private Instant getTimestamp(String timestamp) {
        return DateTimeFormatter.ISO_ZONED_DATE_TIME.parse((CharSequence)timestamp, Instant::from);
    }

    private Set<LocationPoint> detectDistanceJumpAnomalies(List<LocationPoint> points) {
        HashSet<LocationPoint> anomalies = new HashSet<LocationPoint>();
        for (int i = 1; i < points.size(); ++i) {
            double maxDistance;
            LocationPoint prev = points.get(i - 1);
            LocationPoint curr = points.get(i);
            double distance = GeoUtils.distanceInMeters((LocationPoint)prev, (LocationPoint)curr);
            if (this.isEdgePoint(i, points.size())) {
                double d = this.config.maxDistanceJumpMeters;
                Objects.requireNonNull(this.config);
                v1 = d * 1.5;
            } else {
                v1 = maxDistance = this.config.maxDistanceJumpMeters;
            }
            if (!(distance > maxDistance)) continue;
            if (this.isEdgePoint(i, points.size())) {
                anomalies.add(this.selectWorsePoint(prev, curr, points, i));
                continue;
            }
            if (curr.getAccuracyMeters() > prev.getAccuracyMeters()) {
                anomalies.add(curr);
                continue;
            }
            anomalies.add(prev);
        }
        logger.debug("Filtering out [{}] points because distance jumped more than [{}] meters.", (Object)anomalies.size(), (Object)this.config.maxDistanceJumpMeters);
        return anomalies;
    }

    private Set<LocationPoint> detectDirectionAnomalies(List<LocationPoint> points) {
        HashSet<LocationPoint> anomalies = new HashSet<LocationPoint>();
        if (points.size() < 3) {
            return anomalies;
        }
        for (int i = 1; i < points.size() - 1; ++i) {
            LocationPoint prev = points.get(i - 1);
            LocationPoint curr = points.get(i);
            LocationPoint next = points.get(i + 1);
            double bearing1 = this.calculateBearing(prev, curr);
            double bearing2 = this.calculateBearing(curr, next);
            double angleDiff = Math.abs(bearing2 - bearing1);
            if (angleDiff > 180.0) {
                angleDiff = 360.0 - angleDiff;
            }
            double dist1 = GeoUtils.distanceInMeters((LocationPoint)prev, (LocationPoint)curr);
            double dist2 = GeoUtils.distanceInMeters((LocationPoint)next, (LocationPoint)curr);
            if (!(angleDiff > 150.0) || !(dist1 > 50.0) || !(dist2 > 50.0) || !(curr.getAccuracyMeters() > Math.max(prev.getAccuracyMeters(), next.getAccuracyMeters()))) continue;
            anomalies.add(curr);
        }
        logger.debug("Filtering out [{}] points because the suddenly changed the direction.", (Object)anomalies.size());
        return anomalies;
    }

    private LocationPoint selectWorsePoint(LocationPoint p1, LocationPoint p2, List<LocationPoint> allPoints, int currentIndex) {
        double prevPrevDist2;
        LocationPoint prevPrev;
        double prevPrevDist1;
        double dist2Next;
        LocationPoint next;
        double dist1Next;
        double accuracyScore1 = p1.getAccuracyMeters();
        double accuracyScore2 = p2.getAccuracyMeters();
        if (currentIndex == 1 && allPoints.size() > 2 && Math.abs((dist1Next = GeoUtils.distanceInMeters((LocationPoint)p1, (LocationPoint)(next = allPoints.get(currentIndex + 1)))) - (dist2Next = GeoUtils.distanceInMeters((LocationPoint)p2, (LocationPoint)next))) > 1000.0) {
            return dist1Next > dist2Next ? p1 : p2;
        }
        if (currentIndex == allPoints.size() - 1 && allPoints.size() > 2 && Math.abs((prevPrevDist1 = GeoUtils.distanceInMeters((LocationPoint)(prevPrev = allPoints.get(currentIndex - 2)), (LocationPoint)p1)) - (prevPrevDist2 = GeoUtils.distanceInMeters((LocationPoint)prevPrev, (LocationPoint)p2))) > 1000.0) {
            return prevPrevDist1 > prevPrevDist2 ? p1 : p2;
        }
        return accuracyScore1 > accuracyScore2 ? p1 : p2;
    }

    private boolean isEdgePoint(int index, int totalSize) {
        return index == 0 || index == totalSize - 1;
    }

    private double calculateBearing(LocationPoint from, LocationPoint to) {
        double lat1 = Math.toRadians(from.getLatitude());
        double lat2 = Math.toRadians(to.getLatitude());
        double deltaLng = Math.toRadians(to.getLongitude() - from.getLongitude());
        double y = Math.sin(deltaLng) * Math.cos(lat2);
        double x = Math.cos(lat1) * Math.sin(lat2) - Math.sin(lat1) * Math.cos(lat2) * Math.cos(deltaLng);
        double bearing = Math.toDegrees(Math.atan2(y, x));
        return (bearing + 360.0) % 360.0;
    }
}

