/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service.processing;

import com.dedicatedcode.reitti.config.LocationDensityConfig;
import com.dedicatedcode.reitti.dto.LocationPoint;
import com.dedicatedcode.reitti.model.geo.RawLocationPoint;
import com.dedicatedcode.reitti.model.processing.DetectionParameter;
import com.dedicatedcode.reitti.model.security.User;
import com.dedicatedcode.reitti.repository.RawLocationPointJdbcService;
import com.dedicatedcode.reitti.service.VisitDetectionParametersService;
import com.dedicatedcode.reitti.service.processing.LocationDataDensityNormalizer;
import com.dedicatedcode.reitti.service.processing.SyntheticLocationPointGenerator;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class LocationDataDensityNormalizer {
    private static final Logger logger = LoggerFactory.getLogger(LocationDataDensityNormalizer.class);
    private final LocationDensityConfig config;
    private final RawLocationPointJdbcService rawLocationPointService;
    private final SyntheticLocationPointGenerator syntheticGenerator;
    private final VisitDetectionParametersService visitDetectionParametersService;
    private final ConcurrentHashMap<String, ReentrantLock> userLocks = new ConcurrentHashMap();

    @Autowired
    public LocationDataDensityNormalizer(LocationDensityConfig config, RawLocationPointJdbcService rawLocationPointService, SyntheticLocationPointGenerator syntheticGenerator, VisitDetectionParametersService visitDetectionParametersService) {
        this.config = config;
        this.rawLocationPointService = rawLocationPointService;
        this.syntheticGenerator = syntheticGenerator;
        this.visitDetectionParametersService = visitDetectionParametersService;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void normalize(User user, List<LocationPoint> newPoints) {
        if (newPoints == null || newPoints.isEmpty()) {
            logger.trace("No points to normalize for user {}", (Object)user.getUsername());
            return;
        }
        ReentrantLock userLock = this.userLocks.computeIfAbsent(user.getUsername(), string -> new ReentrantLock());
        userLock.lock();
        try {
            logger.debug("Starting batch density normalization for {} points for user {}", (Object)newPoints.size(), (Object)user.getUsername());
            TimeRange inputRange = this.computeTimeRange(newPoints);
            DetectionParameter detectionParams = this.visitDetectionParametersService.getCurrentConfiguration(user, inputRange.start());
            DetectionParameter.LocationDensity densityConfig = detectionParams.getLocationDensity();
            long maxInterpolationGapMinutes = densityConfig.getMaxInterpolationGapMinutes();
            Duration window = Duration.ofMinutes(maxInterpolationGapMinutes);
            TimeRange expandedRange = new TimeRange(inputRange.start().minus(window), inputRange.end().plus(window));
            this.rawLocationPointService.deleteSyntheticPointsInRange(user, expandedRange.start(), expandedRange.end());
            List existingPoints = this.rawLocationPointService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, expandedRange.start().minus(maxInterpolationGapMinutes, ChronoUnit.MINUTES), expandedRange.end().plus(maxInterpolationGapMinutes, ChronoUnit.MINUTES));
            logger.debug("Found {} existing points in expanded range [{} - {}]", new Object[]{existingPoints.size(), expandedRange.start(), expandedRange.end()});
            existingPoints.sort(Comparator.comparing(RawLocationPoint::getTimestamp).thenComparing(p -> p.getGeom().latitude()).thenComparing(p -> p.getGeom().longitude()).thenComparing(RawLocationPoint::isSynthetic));
            logger.trace("Processing {} total points after merge", (Object)existingPoints.size());
            this.processGaps(user, existingPoints, densityConfig);
            this.handleExcessDensity(user, existingPoints);
            logger.debug("Completed batch density normalization for user {}", (Object)user.getUsername());
        }
        catch (Exception e) {
            logger.error("Error during batch density normalization for user {}: {}", new Object[]{user.getUsername(), e.getMessage(), e});
        }
        finally {
            userLock.unlock();
        }
    }

    private TimeRange computeTimeRange(List<LocationPoint> points) {
        Instant minTime = null;
        Instant maxTime = null;
        for (LocationPoint point : points) {
            Instant timestamp = Instant.parse(point.getTimestamp());
            if (minTime == null || timestamp.isBefore(minTime)) {
                minTime = timestamp;
            }
            if (maxTime != null && !timestamp.isAfter(maxTime)) continue;
            maxTime = timestamp;
        }
        return new TimeRange(minTime, maxTime);
    }

    private void processGaps(User user, List<RawLocationPoint> points, DetectionParameter.LocationDensity densityConfig) {
        if (points.size() < 2) {
            return;
        }
        int gapThresholdSeconds = this.config.getGapThresholdSeconds();
        long maxInterpolationSeconds = densityConfig.getMaxInterpolationGapMinutes() * 60L;
        ArrayList allSyntheticPoints = new ArrayList();
        for (int i = 0; i < points.size() - 1; ++i) {
            long gapSeconds;
            RawLocationPoint current = points.get(i);
            RawLocationPoint next = points.get(i + 1);
            if (current.isIgnored() || next.isIgnored() || current.isSynthetic() || (gapSeconds = Duration.between(current.getTimestamp(), next.getTimestamp()).getSeconds()) <= (long)gapThresholdSeconds || gapSeconds > maxInterpolationSeconds) continue;
            List syntheticPoints = this.syntheticGenerator.generateSyntheticPoints(current, next, this.config.getTargetPointsPerMinute(), densityConfig.getMaxInterpolationDistanceMeters());
            logger.trace("Found gap of {} seconds between {} and {} -> created {} synthetic points between them", new Object[]{gapSeconds, current.getTimestamp(), next.getTimestamp(), syntheticPoints.size()});
            allSyntheticPoints.addAll(syntheticPoints);
        }
        if (!allSyntheticPoints.isEmpty()) {
            int inserted = this.rawLocationPointService.bulkInsertSynthetic(user, allSyntheticPoints);
            logger.debug("Inserted {} synthetic points for user {}", (Object)inserted, (Object)user.getUsername());
        }
    }

    private void handleExcessDensity(User user, List<RawLocationPoint> points) {
        if (points.size() < 2) {
            return;
        }
        int toleranceSeconds = this.config.getToleranceSeconds() - 1;
        LinkedHashSet<Long> pointsToIgnore = new LinkedHashSet<Long>();
        HashSet<Long> alreadyConsidered = new HashSet<Long>();
        for (int i = 0; i < points.size() - 1; ++i) {
            RawLocationPoint toIgnore;
            long timeDiff;
            RawLocationPoint current = points.get(i);
            RawLocationPoint next = points.get(i + 1);
            if (current.getId() == null || next.getId() == null || current.isSynthetic() || next.isSynthetic() || current.isIgnored() || next.isIgnored() || alreadyConsidered.contains(current.getId()) || alreadyConsidered.contains(next.getId()) || (timeDiff = Duration.between(current.getTimestamp(), next.getTimestamp()).getSeconds()) >= (long)toleranceSeconds || (toIgnore = this.selectPointToIgnore(current, next)) == null || toIgnore.getId() == null) continue;
            pointsToIgnore.add(toIgnore.getId());
            alreadyConsidered.add(toIgnore.getId());
            logger.trace("Marking point {} as ignored due to excess density", (Object)toIgnore.getId());
        }
        if (!pointsToIgnore.isEmpty()) {
            this.rawLocationPointService.bulkUpdateIgnoredStatus(new ArrayList(pointsToIgnore), true);
            logger.debug("Marked {} points as ignored for user {}", (Object)pointsToIgnore.size(), (Object)user.getUsername());
        }
    }

    private RawLocationPoint selectPointToIgnore(RawLocationPoint point1, RawLocationPoint point2) {
        int timestampCompare;
        if (!point1.isSynthetic() && point2.isSynthetic()) {
            return point2;
        }
        if (point1.isSynthetic() && !point2.isSynthetic()) {
            return point1;
        }
        Double acc1 = point1.getAccuracyMeters();
        Double acc2 = point2.getAccuracyMeters();
        if (acc1 != null && acc2 != null) {
            if (!acc1.equals(acc2)) {
                return acc1 < acc2 ? point2 : point1;
            }
        } else {
            if (acc1 != null) {
                return point2;
            }
            if (acc2 != null) {
                return point1;
            }
        }
        if ((timestampCompare = point1.getTimestamp().compareTo(point2.getTimestamp())) != 0) {
            return timestampCompare < 0 ? point2 : point1;
        }
        int latCompare = Double.compare(point1.getGeom().latitude(), point2.getGeom().latitude());
        if (latCompare != 0) {
            return latCompare < 0 ? point2 : point1;
        }
        int lonCompare = Double.compare(point1.getGeom().longitude(), point2.getGeom().longitude());
        return lonCompare < 0 ? point2 : point1;
    }
}

