/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service.processing;

import com.dedicatedcode.reitti.dto.LocationPoint;
import com.dedicatedcode.reitti.model.geo.GeoPoint;
import com.dedicatedcode.reitti.model.geo.GeoUtils;
import com.dedicatedcode.reitti.model.geo.RawLocationPoint;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class SyntheticLocationPointGenerator {
    private static final Logger logger = LoggerFactory.getLogger(SyntheticLocationPointGenerator.class);

    public List<LocationPoint> generateSyntheticPoints(RawLocationPoint startPoint, RawLocationPoint endPoint, int targetPointsPerMinute, double maxDistanceMeters) {
        if (!this.shouldInterpolate(startPoint, endPoint, maxDistanceMeters)) {
            logger.trace("Skipping interpolation between points: distance too large or other constraints not met");
            return List.of();
        }
        ArrayList<LocationPoint> syntheticPoints = new ArrayList<LocationPoint>();
        int intervalSeconds = 60 / targetPointsPerMinute;
        Instant startTime = startPoint.getTimestamp();
        Instant endTime = endPoint.getTimestamp();
        Instant currentTime = startTime.plusSeconds(intervalSeconds).truncatedTo(ChronoUnit.SECONDS);
        while (currentTime.isBefore(endTime)) {
            long totalDuration = endTime.getEpochSecond() - startTime.getEpochSecond();
            long currentDuration = currentTime.getEpochSecond() - startTime.getEpochSecond();
            double timeRatio = (double)currentDuration / (double)totalDuration;
            double distance = GeoUtils.distanceInMeters((RawLocationPoint)startPoint, (RawLocationPoint)endPoint);
            double speed = distance / (double)totalDuration;
            double speedFactor = Math.min(speed / 5.0, 1.0);
            double ratio = Math.pow(timeRatio, 1.0 / (speedFactor + 0.5));
            GeoPoint interpolatedCoords = this.interpolateCoordinates(startPoint.getGeom(), endPoint.getGeom(), ratio);
            Double interpolatedAccuracy = this.interpolateValue(startPoint.getAccuracyMeters(), endPoint.getAccuracyMeters(), ratio);
            Double interpolatedElevation = this.interpolateValue(startPoint.getElevationMeters(), endPoint.getElevationMeters(), ratio);
            LocationPoint syntheticPoint = new LocationPoint();
            syntheticPoint.setLatitude(Double.valueOf(interpolatedCoords.latitude()));
            syntheticPoint.setLongitude(Double.valueOf(interpolatedCoords.longitude()));
            syntheticPoint.setTimestamp(currentTime.atOffset(ZoneOffset.UTC).format(DateTimeFormatter.ISO_OFFSET_DATE_TIME));
            syntheticPoint.setAccuracyMeters(interpolatedAccuracy);
            syntheticPoint.setElevationMeters(interpolatedElevation);
            syntheticPoints.add(syntheticPoint);
            currentTime = currentTime.plusSeconds(intervalSeconds);
        }
        logger.trace("Generated {} synthetic points between {} and {}", new Object[]{syntheticPoints.size(), startTime, endTime});
        return syntheticPoints;
    }

    private boolean shouldInterpolate(RawLocationPoint start, RawLocationPoint end, double maxDistance) {
        double distance = GeoUtils.distanceInMeters((RawLocationPoint)start, (RawLocationPoint)end);
        if (distance > maxDistance) {
            logger.trace("Distance {} meters exceeds maximum interpolation distance {} meters", (Object)distance, (Object)maxDistance);
            return false;
        }
        return true;
    }

    private GeoPoint interpolateCoordinates(GeoPoint start, GeoPoint end, double ratio) {
        double lat = start.latitude() + (end.latitude() - start.latitude()) * ratio;
        double lon = start.longitude() + (end.longitude() - start.longitude()) * ratio;
        return new GeoPoint(lat, lon);
    }

    private Double interpolateValue(Double start, Double end, double ratio) {
        if (start == null && end == null) {
            return null;
        }
        if (start == null) {
            return end;
        }
        if (end == null) {
            return start;
        }
        return start + (end - start) * ratio;
    }
}

