/*
 * Decompiled with CFR 0.152.
 */
package com.dedicatedcode.reitti.service.processing;

import com.dedicatedcode.reitti.event.LocationProcessEvent;
import com.dedicatedcode.reitti.event.SignificantPlaceCreatedEvent;
import com.dedicatedcode.reitti.model.ClusteredPoint;
import com.dedicatedcode.reitti.model.PlaceInformationOverride;
import com.dedicatedcode.reitti.model.geo.GeoPoint;
import com.dedicatedcode.reitti.model.geo.GeoUtils;
import com.dedicatedcode.reitti.model.geo.ProcessedVisit;
import com.dedicatedcode.reitti.model.geo.RawLocationPoint;
import com.dedicatedcode.reitti.model.geo.SignificantPlace;
import com.dedicatedcode.reitti.model.geo.StayPoint;
import com.dedicatedcode.reitti.model.geo.TransportMode;
import com.dedicatedcode.reitti.model.geo.Trip;
import com.dedicatedcode.reitti.model.geo.Visit;
import com.dedicatedcode.reitti.model.processing.DetectionParameter;
import com.dedicatedcode.reitti.model.security.User;
import com.dedicatedcode.reitti.repository.PreviewProcessedVisitJdbcService;
import com.dedicatedcode.reitti.repository.PreviewRawLocationPointJdbcService;
import com.dedicatedcode.reitti.repository.PreviewSignificantPlaceJdbcService;
import com.dedicatedcode.reitti.repository.PreviewTripJdbcService;
import com.dedicatedcode.reitti.repository.PreviewVisitDetectionParametersJdbcService;
import com.dedicatedcode.reitti.repository.ProcessedVisitJdbcService;
import com.dedicatedcode.reitti.repository.RawLocationPointJdbcService;
import com.dedicatedcode.reitti.repository.SignificantPlaceJdbcService;
import com.dedicatedcode.reitti.repository.SignificantPlaceOverrideJdbcService;
import com.dedicatedcode.reitti.repository.TripJdbcService;
import com.dedicatedcode.reitti.repository.UserJdbcService;
import com.dedicatedcode.reitti.service.GeoLocationTimezoneService;
import com.dedicatedcode.reitti.service.UserNotificationService;
import com.dedicatedcode.reitti.service.VisitDetectionParametersService;
import com.dedicatedcode.reitti.service.processing.TransportModeService;
import com.dedicatedcode.reitti.service.processing.UnifiedLocationProcessingService;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.stereotype.Service;

@Service
public class UnifiedLocationProcessingService {
    private static final Logger logger = LoggerFactory.getLogger(UnifiedLocationProcessingService.class);
    private final UserJdbcService userJdbcService;
    private final RawLocationPointJdbcService rawLocationPointJdbcService;
    private final PreviewRawLocationPointJdbcService previewRawLocationPointJdbcService;
    private final ProcessedVisitJdbcService processedVisitJdbcService;
    private final PreviewProcessedVisitJdbcService previewProcessedVisitJdbcService;
    private final TripJdbcService tripJdbcService;
    private final PreviewTripJdbcService previewTripJdbcService;
    private final SignificantPlaceJdbcService significantPlaceJdbcService;
    private final PreviewSignificantPlaceJdbcService previewSignificantPlaceJdbcService;
    private final SignificantPlaceOverrideJdbcService significantPlaceOverrideJdbcService;
    private final VisitDetectionParametersService visitDetectionParametersService;
    private final PreviewVisitDetectionParametersJdbcService previewVisitDetectionParametersJdbcService;
    private final TransportModeService transportModeService;
    private final UserNotificationService userNotificationService;
    private final GeoLocationTimezoneService timezoneService;
    private final GeometryFactory geometryFactory;
    private final RabbitTemplate rabbitTemplate;

    public UnifiedLocationProcessingService(UserJdbcService userJdbcService, RawLocationPointJdbcService rawLocationPointJdbcService, PreviewRawLocationPointJdbcService previewRawLocationPointJdbcService, ProcessedVisitJdbcService processedVisitJdbcService, PreviewProcessedVisitJdbcService previewProcessedVisitJdbcService, TripJdbcService tripJdbcService, PreviewTripJdbcService previewTripJdbcService, SignificantPlaceJdbcService significantPlaceJdbcService, PreviewSignificantPlaceJdbcService previewSignificantPlaceJdbcService, SignificantPlaceOverrideJdbcService significantPlaceOverrideJdbcService, VisitDetectionParametersService visitDetectionParametersService, PreviewVisitDetectionParametersJdbcService previewVisitDetectionParametersJdbcService, TransportModeService transportModeService, UserNotificationService userNotificationService, GeoLocationTimezoneService timezoneService, GeometryFactory geometryFactory, RabbitTemplate rabbitTemplate) {
        this.userJdbcService = userJdbcService;
        this.rawLocationPointJdbcService = rawLocationPointJdbcService;
        this.previewRawLocationPointJdbcService = previewRawLocationPointJdbcService;
        this.processedVisitJdbcService = processedVisitJdbcService;
        this.previewProcessedVisitJdbcService = previewProcessedVisitJdbcService;
        this.tripJdbcService = tripJdbcService;
        this.previewTripJdbcService = previewTripJdbcService;
        this.significantPlaceJdbcService = significantPlaceJdbcService;
        this.previewSignificantPlaceJdbcService = previewSignificantPlaceJdbcService;
        this.significantPlaceOverrideJdbcService = significantPlaceOverrideJdbcService;
        this.visitDetectionParametersService = visitDetectionParametersService;
        this.previewVisitDetectionParametersJdbcService = previewVisitDetectionParametersJdbcService;
        this.transportModeService = transportModeService;
        this.userNotificationService = userNotificationService;
        this.timezoneService = timezoneService;
        this.geometryFactory = geometryFactory;
        this.rabbitTemplate = rabbitTemplate;
    }

    public void processLocationEvent(LocationProcessEvent event) {
        long startTime = System.currentTimeMillis();
        String username = event.getUsername();
        String previewId = event.getPreviewId();
        logger.info("Processing location data for user [{}], mode: {}", (Object)username, (Object)(previewId == null ? "LIVE" : "PREVIEW"));
        User user = (User)this.userJdbcService.findByUsername(username).orElseThrow(() -> new IllegalStateException("User not found: " + username));
        VisitDetectionResult detectionResult = this.detectVisits(user, event);
        logger.debug("Detection: {} visits created", (Object)detectionResult.visits.size());
        VisitMergingResult mergingResult = this.mergeVisits(user, previewId, event.getTraceId(), detectionResult.searchStart, detectionResult.searchEnd, detectionResult.visits);
        logger.debug("Merging: {} visits merged into {} processed visits", (Object)mergingResult.inputVisits.size(), (Object)mergingResult.processedVisits.size());
        TripDetectionResult tripResult = this.detectTrips(user, previewId, mergingResult.searchStart, mergingResult.searchEnd, mergingResult.processedVisits);
        logger.debug("Trip detection: {} trips created", (Object)tripResult.trips.size());
        if (previewId == null) {
            this.userNotificationService.newVisits(user, mergingResult.processedVisits);
            this.userNotificationService.newTrips(user, tripResult.trips);
        } else {
            this.userNotificationService.newTrips(user, tripResult.trips, previewId);
        }
        long duration = System.currentTimeMillis() - startTime;
        if (logger.isTraceEnabled()) {
            StringBuilder traceOutput = new StringBuilder();
            traceOutput.append("\n=== PROCESSING RESULTS FOR USER [").append(username).append("] ===\n");
            traceOutput.append("Event Period: ").append(event.getEarliest()).append(" \u2192 ").append(event.getLatest()).append("\n");
            traceOutput.append("Search Period: ").append(detectionResult.searchStart).append(" \u2192 ").append(detectionResult.searchEnd).append("\n");
            traceOutput.append("Duration: ").append(duration).append("ms\n\n");
            traceOutput.append("INPUT VISITS (").append(detectionResult.visits.size()).append(") - took [").append(detectionResult.durationInMillis).append("]ms:\n");
            traceOutput.append("\u250c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2510\n");
            traceOutput.append("\u2502 Start Time          \u2502 End Time            \u2502 Duration  \u2502 Latitude    \u2502 Longitude   \u2502 Google Maps Link                                                     \u2502\n");
            traceOutput.append("\u251c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2524\n");
            for (Visit visit : detectionResult.visits) {
                String googleMapsLink = "https://www.google.com/maps/search/?api=1&query=" + visit.getLatitude() + "," + visit.getLongitude();
                traceOutput.append(String.format("\u2502 %-19s \u2502 %-19s \u2502 %8ds \u2502 %11.6f \u2502 %11.6f \u2502 %-68s \u2502\n", visit.getStartTime().toString().substring(0, 19), visit.getEndTime().toString().substring(0, 19), visit.getDurationSeconds(), visit.getLatitude(), visit.getLongitude(), googleMapsLink));
            }
            traceOutput.append("\u2514\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2518\n\n");
            traceOutput.append("PROCESSED VISITS (").append(mergingResult.processedVisits.size()).append(") - took [").append(mergingResult.durationInMillis).append("]ms:\n");
            traceOutput.append("\u250c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2510\n");
            traceOutput.append("\u2502 Start Time          \u2502 End Time            \u2502 Duration  \u2502 Latitude    \u2502 Longitude   \u2502 Place Name           \u2502\n");
            traceOutput.append("\u251c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2524\n");
            for (Visit visit : mergingResult.processedVisits) {
                Object placeName;
                Object object = placeName = visit.getPlace().getName() != null ? visit.getPlace().getName() : "Unnamed Place";
                if (((String)placeName).length() > 20) {
                    placeName = ((String)placeName).substring(0, 17) + "...";
                }
                traceOutput.append(String.format("\u2502 %-19s \u2502 %-19s \u2502 %8ds \u2502 %11.6f \u2502 %11.6f \u2502 %-20s \u2502\n", visit.getStartTime().toString().substring(0, 19), visit.getEndTime().toString().substring(0, 19), visit.getDurationSeconds(), visit.getPlace().getLatitudeCentroid(), visit.getPlace().getLongitudeCentroid(), placeName));
            }
            traceOutput.append("\u2514\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2518\n\n");
            traceOutput.append("TRIPS (").append(tripResult.trips.size()).append(") - took [").append(tripResult.durationInMillis).append("]ms:\n");
            traceOutput.append("\u250c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u252c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2510\n");
            traceOutput.append("\u2502 Start Time          \u2502 End Time            \u2502 Duration  \u2502 Distance  \u2502 Traveled  \u2502 Transport Mode  \u2502\n");
            traceOutput.append("\u251c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u253c\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2524\n");
            for (Trip trip : tripResult.trips) {
                traceOutput.append(String.format("\u2502 %-19s \u2502 %-19s \u2502 %8ds \u2502 %8.0fm \u2502 %8.0fm \u2502 %-15s \u2502\n", trip.getStartTime().toString().substring(0, 19), trip.getEndTime().toString().substring(0, 19), trip.getDurationSeconds(), trip.getEstimatedDistanceMeters(), trip.getTravelledDistanceMeters(), trip.getTransportModeInferred().toString()));
            }
            traceOutput.append("\u2514\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2534\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2500\u2518\n");
            logger.trace(traceOutput.toString());
        }
        logger.info("Completed processing for user [{}] in {}ms: {} visits \u2192 {} processed visits \u2192 {} trips", new Object[]{username, duration, detectionResult.visits.size(), mergingResult.processedVisits.size(), tripResult.trips.size()});
    }

    private VisitDetectionResult detectVisits(User user, LocationProcessEvent event) {
        long start = System.currentTimeMillis();
        String previewId = event.getPreviewId();
        Instant windowStart = event.getEarliest().minus(1L, ChronoUnit.DAYS);
        Instant windowEnd = event.getLatest().plus(1L, ChronoUnit.DAYS);
        DetectionParameter currentConfiguration = previewId == null ? this.visitDetectionParametersService.getCurrentConfiguration(user, windowStart) : this.previewVisitDetectionParametersJdbcService.findCurrent(user, previewId);
        DetectionParameter.VisitDetection detectionParams = currentConfiguration.getVisitDetection();
        List existingProcessedVisits = previewId == null ? this.processedVisitJdbcService.findByUserAndStartTimeBeforeEqualAndEndTimeAfterEqual(user, windowEnd, windowStart) : this.previewProcessedVisitJdbcService.findByUserAndStartTimeBeforeEqualAndEndTimeAfterEqual(user, previewId, windowEnd, windowStart);
        if (!existingProcessedVisits.isEmpty()) {
            if (((ProcessedVisit)existingProcessedVisits.getFirst()).getStartTime().isBefore(windowStart)) {
                windowStart = ((ProcessedVisit)existingProcessedVisits.getFirst()).getStartTime();
            }
            if (((ProcessedVisit)existingProcessedVisits.getLast()).getEndTime().isAfter(windowEnd)) {
                windowEnd = ((ProcessedVisit)existingProcessedVisits.getLast()).getEndTime();
            }
        }
        double baseLatitude = existingProcessedVisits.isEmpty() ? 50.0 : ((ProcessedVisit)existingProcessedVisits.getFirst()).getPlace().getLatitudeCentroid();
        double metersAsDegrees = GeoUtils.metersToDegreesAtPosition((double)((double)currentConfiguration.getVisitMerging().getMinDistanceBetweenVisits() / 4.0), (double)baseLatitude);
        int minimumAdjacentPoints = Math.max(4, Math.toIntExact(detectionParams.getMinimumStayTimeInSeconds() / 60L));
        List clusteredPoints = previewId == null ? this.rawLocationPointJdbcService.findClusteredPointsInTimeRangeForUser(user, windowStart, windowEnd, minimumAdjacentPoints, metersAsDegrees) : this.previewRawLocationPointJdbcService.findClusteredPointsInTimeRangeForUser(user, previewId, windowStart, windowEnd, minimumAdjacentPoints, metersAsDegrees);
        logger.debug("Searching for clustered points in range [{}, {}], minimum adjacent points: {} ", new Object[]{windowStart, windowEnd, minimumAdjacentPoints});
        TreeMap<Integer, List> clusteredByLocation = new TreeMap<Integer, List>();
        for (ClusteredPoint cp : clusteredPoints.stream().filter(clusteredPoint -> !clusteredPoint.getPoint().isIgnored()).toList()) {
            if (cp.getClusterId() == null) continue;
            clusteredByLocation.computeIfAbsent(cp.getClusterId(), n -> new ArrayList()).add(cp.getPoint());
        }
        List stayPoints = this.detectStayPointsFromTrajectory(clusteredByLocation, detectionParams);
        List<Visit> visits = stayPoints.stream().map(sp -> new Visit(Double.valueOf(sp.getLongitude()), Double.valueOf(sp.getLatitude()), sp.getArrivalTime(), sp.getDepartureTime(), Long.valueOf(sp.getDurationSeconds()), false)).sorted(Comparator.comparing(Visit::getStartTime)).toList();
        return new VisitDetectionResult(visits, windowStart, windowEnd, System.currentTimeMillis() - start);
    }

    private VisitMergingResult mergeVisits(User user, String previewId, String traceId, Instant initialStart, Instant initialEnd, List<Visit> allVisits) {
        List existingProcessedVisits;
        long start = System.currentTimeMillis();
        DetectionParameter.VisitMerging mergeConfig = previewId == null ? this.visitDetectionParametersService.getCurrentConfiguration(user, initialStart).getVisitMerging() : this.previewVisitDetectionParametersJdbcService.findCurrent(user, previewId).getVisitMerging();
        Instant searchStart = initialStart;
        Instant searchEnd = initialEnd;
        if (previewId == null) {
            existingProcessedVisits = this.processedVisitJdbcService.findByUserAndStartTimeBeforeEqualAndEndTimeAfterEqual(user, searchEnd, searchStart);
            this.processedVisitJdbcService.deleteAll(existingProcessedVisits);
        } else {
            existingProcessedVisits = this.previewProcessedVisitJdbcService.findByUserAndStartTimeBeforeEqualAndEndTimeAfterEqual(user, previewId, searchEnd, searchStart);
            this.previewProcessedVisitJdbcService.deleteAll(existingProcessedVisits);
        }
        if (!existingProcessedVisits.isEmpty()) {
            if (((ProcessedVisit)existingProcessedVisits.getFirst()).getStartTime().isBefore(searchStart)) {
                searchStart = ((ProcessedVisit)existingProcessedVisits.getFirst()).getStartTime();
            }
            if (((ProcessedVisit)existingProcessedVisits.getLast()).getEndTime().isAfter(searchEnd)) {
                searchEnd = ((ProcessedVisit)existingProcessedVisits.getLast()).getEndTime();
            }
        }
        if (allVisits.isEmpty()) {
            return new VisitMergingResult(List.of(), List.of(), searchStart, searchEnd, System.currentTimeMillis() - start);
        }
        List processedVisits = this.mergeVisitsChronologically(user, previewId, traceId, allVisits, mergeConfig);
        processedVisits = previewId == null ? this.processedVisitJdbcService.bulkInsert(user, processedVisits) : this.previewProcessedVisitJdbcService.bulkInsert(user, previewId, processedVisits);
        return new VisitMergingResult(allVisits, processedVisits, searchStart, searchEnd, System.currentTimeMillis() - start);
    }

    private TripDetectionResult detectTrips(User user, String previewId, Instant searchStart, Instant searchEnd, List<ProcessedVisit> processedVisits) {
        long start = System.currentTimeMillis();
        processedVisits.sort(Comparator.comparing(ProcessedVisit::getStartTime));
        if (previewId == null) {
            existingTrips = this.tripJdbcService.findByUserAndTimeOverlap(user, searchStart, searchEnd);
            this.tripJdbcService.deleteAll(existingTrips);
        } else {
            existingTrips = this.previewTripJdbcService.findByUserAndTimeOverlap(user, previewId, searchStart, searchEnd);
            this.previewTripJdbcService.deleteAll(existingTrips);
        }
        List<Trip> trips = new ArrayList<Trip>();
        for (int i = 0; i < processedVisits.size() - 1; ++i) {
            ProcessedVisit endVisit;
            ProcessedVisit startVisit = processedVisits.get(i);
            Trip trip = this.createTripBetweenVisits(user, previewId, startVisit, endVisit = processedVisits.get(i + 1));
            if (trip == null) continue;
            trips.add(trip);
        }
        if (previewId == null) {
            Optional processedVisitAfter;
            Optional firstProcessedVisitBefore = this.processedVisitJdbcService.findFirstProcessedVisitBefore(user, searchStart);
            if (firstProcessedVisitBefore.isPresent() && Duration.between(((ProcessedVisit)firstProcessedVisitBefore.get()).getEndTime(), processedVisits.getFirst().getStartTime()).toHours() <= 24L) {
                trips.add(this.createTripBetweenVisits(user, null, (ProcessedVisit)firstProcessedVisitBefore.get(), processedVisits.getFirst()));
            }
            if ((processedVisitAfter = this.processedVisitJdbcService.findFirstProcessedVisitAfter(user, searchEnd)).isPresent() && Duration.between(processedVisits.getLast().getEndTime(), ((ProcessedVisit)processedVisitAfter.get()).getStartTime()).toHours() <= 24L) {
                trips.add(this.createTripBetweenVisits(user, null, processedVisits.getLast(), (ProcessedVisit)processedVisitAfter.get()));
            }
        }
        trips.sort(Comparator.comparing(Trip::getStartTime));
        trips = previewId == null ? this.tripJdbcService.bulkInsert(user, trips) : this.previewTripJdbcService.bulkInsert(user, previewId, trips);
        return new TripDetectionResult(trips, System.currentTimeMillis() - start);
    }

    private List<StayPoint> detectStayPointsFromTrajectory(Map<Integer, List<RawLocationPoint>> points, DetectionParameter.VisitDetection visitDetectionParameters) {
        logger.debug("Starting cluster-based stay point detection with {} different spatial clusters.", (Object)points.size());
        ArrayList clusters = new ArrayList();
        for (List<RawLocationPoint> clusteredByLocation : points.values()) {
            logger.debug("Start splitting up geospatial cluster with [{}] elements based on minimum time [{}]s between points", (Object)clusteredByLocation.size(), (Object)visitDetectionParameters.getMaxMergeTimeBetweenSameStayPoints());
            clusteredByLocation.sort(Comparator.comparing(RawLocationPoint::getTimestamp));
            ArrayList<RawLocationPoint> currentTimedCluster = new ArrayList<RawLocationPoint>();
            clusters.add(currentTimedCluster);
            currentTimedCluster.add(clusteredByLocation.getFirst());
            Instant currentTime = clusteredByLocation.getFirst().getTimestamp();
            for (int i = 1; i < clusteredByLocation.size(); ++i) {
                RawLocationPoint next = clusteredByLocation.get(i);
                if (Duration.between(currentTime, next.getTimestamp()).getSeconds() < visitDetectionParameters.getMaxMergeTimeBetweenSameStayPoints()) {
                    currentTimedCluster.add(next);
                } else {
                    currentTimedCluster = new ArrayList();
                    currentTimedCluster.add(next);
                    clusters.add(currentTimedCluster);
                }
                currentTime = next.getTimestamp();
            }
        }
        logger.debug("Detected {} stay points after splitting them up.", (Object)clusters.size());
        List<List> filteredByMinimumDuration = clusters.stream().filter(c -> Duration.between(((RawLocationPoint)c.getFirst()).getTimestamp(), ((RawLocationPoint)c.getLast()).getTimestamp()).toSeconds() > visitDetectionParameters.getMinimumStayTimeInSeconds()).toList();
        logger.debug("Found {} valid clusters after duration filtering with minimum stay time [{}]s", (Object)filteredByMinimumDuration.size(), (Object)visitDetectionParameters.getMinimumStayTimeInSeconds());
        return filteredByMinimumDuration.stream().map(arg_0 -> this.createStayPoint(arg_0)).collect(Collectors.toList());
    }

    private List<ProcessedVisit> mergeVisitsChronologically(User user, String previewId, String traceId, List<Visit> visits, DetectionParameter.VisitMerging mergeConfiguration) {
        if (visits.isEmpty()) {
            return new ArrayList<ProcessedVisit>();
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Merging [{}] visits between [{}] and [{}]", new Object[]{visits.size(), visits.getFirst().getStartTime(), visits.getLast().getEndTime()});
        }
        ArrayList<ProcessedVisit> result = new ArrayList<ProcessedVisit>();
        Visit currentVisit = visits.getFirst();
        Instant currentStartTime = currentVisit.getStartTime();
        Instant currentEndTime = currentVisit.getEndTime();
        SignificantPlace currentPlace = this.findOrCreateSignificantPlace(user, previewId, currentVisit.getLatitude().doubleValue(), currentVisit.getLongitude().doubleValue(), mergeConfiguration, traceId);
        for (int i = 1; i < visits.size(); ++i) {
            boolean shouldMergeWithNextVisit;
            Visit nextVisit = visits.get(i);
            if (nextVisit.getStartTime().isBefore(currentEndTime)) {
                logger.error("Skipping visit [{}] because it starts before the end time of the previous one [{}]", (Object)nextVisit, (Object)currentEndTime);
                continue;
            }
            SignificantPlace nextPlace = this.findOrCreateSignificantPlace(user, previewId, nextVisit.getLatitude().doubleValue(), nextVisit.getLongitude().doubleValue(), mergeConfiguration, traceId);
            boolean samePlace = nextPlace.getId().equals(currentPlace.getId());
            boolean withinTimeThreshold = Duration.between(currentEndTime, nextVisit.getStartTime()).getSeconds() <= mergeConfiguration.getMaxMergeTimeBetweenSameVisits();
            boolean bl = shouldMergeWithNextVisit = samePlace && withinTimeThreshold;
            if (samePlace && !withinTimeThreshold) {
                List pointsBetweenVisits = previewId == null ? this.rawLocationPointJdbcService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, currentEndTime, nextVisit.getStartTime()) : this.previewRawLocationPointJdbcService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, previewId, currentEndTime, nextVisit.getStartTime());
                if (pointsBetweenVisits.size() > 2) {
                    double travelledDistanceInMeters = GeoUtils.calculateTripDistance((List)pointsBetweenVisits);
                    shouldMergeWithNextVisit = travelledDistanceInMeters <= (double)mergeConfiguration.getMinDistanceBetweenVisits();
                } else {
                    logger.debug("There are no points tracked between {} and {}. Will merge consecutive visits because they are on the same place", (Object)currentEndTime, (Object)nextVisit.getStartTime());
                    shouldMergeWithNextVisit = true;
                }
            }
            if (shouldMergeWithNextVisit) {
                currentEndTime = nextVisit.getEndTime().isAfter(currentEndTime) ? nextVisit.getEndTime() : currentEndTime;
                continue;
            }
            ProcessedVisit processedVisit = this.createProcessedVisit(currentPlace, currentStartTime, currentEndTime);
            if (processedVisit == null) continue;
            result.add(processedVisit);
            Instant previousProcessedVisitEndTime = processedVisit.getEndTime();
            currentPlace = nextPlace;
            currentStartTime = nextVisit.getStartTime();
            currentEndTime = nextVisit.getEndTime();
            if (currentStartTime.isBefore(previousProcessedVisitEndTime)) {
                currentStartTime = previousProcessedVisitEndTime;
            }
            if (!currentEndTime.isBefore(currentStartTime)) continue;
            currentEndTime = currentStartTime;
        }
        ProcessedVisit lastProcessedVisit = this.createProcessedVisit(currentPlace, currentStartTime, currentEndTime);
        if (lastProcessedVisit != null) {
            result.add(lastProcessedVisit);
        }
        return result;
    }

    private ProcessedVisit createProcessedVisit(SignificantPlace place, Instant startTime, Instant endTime) {
        if (endTime.isBefore(startTime)) {
            logger.warn("Skipping zero or negative duration processed visit for place [{}] between [{}] and [{}]", new Object[]{place.getId(), startTime, endTime});
            return null;
        }
        if (endTime.equals(startTime)) {
            logger.warn("Skipping zero duration processed visit for place [{}] from [{} -> {}]", new Object[]{place.getId(), startTime, endTime});
            return null;
        }
        logger.debug("Creating processed visit for place [{}] between [{}] and [{}]", new Object[]{place.getId(), startTime, endTime});
        return new ProcessedVisit(place, startTime, endTime, Long.valueOf(endTime.getEpochSecond() - startTime.getEpochSecond()));
    }

    private StayPoint createStayPoint(List<RawLocationPoint> clusterPoints) {
        GeoPoint result = this.weightedCenter(clusterPoints);
        Instant arrivalTime = clusterPoints.getFirst().getTimestamp();
        Instant departureTime = clusterPoints.getLast().getTimestamp();
        logger.trace("Creating stay point at [{}] with arrival time [{}] and departure time [{}]", new Object[]{result, arrivalTime, departureTime});
        return new StayPoint(result.latitude(), result.longitude(), arrivalTime, departureTime, clusterPoints);
    }

    private GeoPoint weightedCenter(List<RawLocationPoint> clusterPoints) {
        long start = System.currentTimeMillis();
        GeoPoint result = clusterPoints.size() <= 100 ? this.weightedCenterSimple(clusterPoints) : this.weightedCenterOptimized(clusterPoints);
        logger.trace("Weighted center calculation took {}ms for [{}] number of points", (Object)(System.currentTimeMillis() - start), (Object)clusterPoints.size());
        return result;
    }

    private GeoPoint weightedCenterSimple(List<RawLocationPoint> clusterPoints) {
        RawLocationPoint bestPoint = null;
        double maxDensityScore = 0.0;
        for (RawLocationPoint candidate : clusterPoints) {
            double densityScore = 0.0;
            for (RawLocationPoint neighbor : clusterPoints) {
                if (candidate == neighbor) continue;
                double distance = GeoUtils.distanceInMeters((RawLocationPoint)candidate, (RawLocationPoint)neighbor);
                double d = candidate.getAccuracyMeters() != null && candidate.getAccuracyMeters() > 0.0 ? candidate.getAccuracyMeters() : 50.0;
                double accuracy = d;
                if (!(distance <= accuracy * 2.0)) continue;
                double proximityWeight = Math.max(0.0, 1.0 - distance / (accuracy * 2.0));
                double accuracyWeight = 1.0 / accuracy;
                densityScore += proximityWeight * accuracyWeight;
            }
            double d = candidate.getAccuracyMeters() != null && candidate.getAccuracyMeters() > 0.0 ? candidate.getAccuracyMeters() : 50.0;
            if (!((densityScore += 1.0 / d) > maxDensityScore)) continue;
            maxDensityScore = densityScore;
            bestPoint = candidate;
        }
        if (bestPoint == null) {
            bestPoint = clusterPoints.getFirst();
        }
        return new GeoPoint(bestPoint.getLatitude().doubleValue(), bestPoint.getLongitude().doubleValue());
    }

    private GeoPoint weightedCenterOptimized(List<RawLocationPoint> clusterPoints) {
        int sampleSize = Math.min(200, clusterPoints.size());
        List<Object> samplePoints = new ArrayList();
        if (clusterPoints.size() <= sampleSize) {
            samplePoints = clusterPoints;
        } else {
            int step = clusterPoints.size() / sampleSize;
            for (int i = 0; i < clusterPoints.size(); i += step) {
                samplePoints.add(clusterPoints.get(i));
            }
        }
        double minLat = clusterPoints.stream().mapToDouble(RawLocationPoint::getLatitude).min().orElse(0.0);
        double minLon = clusterPoints.stream().mapToDouble(RawLocationPoint::getLongitude).min().orElse(0.0);
        double cellSizeLat = 1.0E-4;
        double cellSizeLon = 1.0E-4;
        HashMap<String, List> grid = new HashMap<String, List>();
        for (RawLocationPoint point : clusterPoints) {
            int gridLat = (int)((point.getLatitude() - minLat) / cellSizeLat);
            int gridLon = (int)((point.getLongitude() - minLon) / cellSizeLon);
            String string2 = gridLat + "," + gridLon;
            grid.computeIfAbsent(string2, string -> new ArrayList()).add(point);
        }
        RawLocationPoint bestPoint = null;
        double maxDensityScore = 0.0;
        for (RawLocationPoint rawLocationPoint : samplePoints) {
            double accuracy = rawLocationPoint.getAccuracyMeters() != null && rawLocationPoint.getAccuracyMeters() > 0.0 ? rawLocationPoint.getAccuracyMeters() : 50.0;
            int candidateGridLat = (int)((rawLocationPoint.getLatitude() - minLat) / cellSizeLat);
            int candidateGridLon = (int)((rawLocationPoint.getLongitude() - minLon) / cellSizeLon);
            int searchRadiusInCells = Math.max(1, (int)(accuracy / 100000.0));
            double densityScore = 0.0;
            for (int latOffset = -searchRadiusInCells; latOffset <= searchRadiusInCells; ++latOffset) {
                for (int lonOffset = -searchRadiusInCells; lonOffset <= searchRadiusInCells; ++lonOffset) {
                    String neighborKey = candidateGridLat + latOffset + "," + (candidateGridLon + lonOffset);
                    List neighbors = (List)grid.get(neighborKey);
                    if (neighbors == null) continue;
                    for (RawLocationPoint neighbor : neighbors) {
                        if (rawLocationPoint == neighbor) continue;
                        double gridDistance = Math.sqrt(latOffset * latOffset + lonOffset * lonOffset);
                        double proximityWeight = Math.max(0.0, 1.0 - gridDistance / (double)searchRadiusInCells);
                        densityScore += proximityWeight;
                    }
                }
            }
            double accuracyWeight = 1.0 / accuracy;
            if (!((densityScore = densityScore * accuracyWeight + accuracyWeight) > maxDensityScore)) continue;
            maxDensityScore = densityScore;
            bestPoint = rawLocationPoint;
        }
        if (bestPoint == null) {
            bestPoint = clusterPoints.getFirst();
        }
        return new GeoPoint(bestPoint.getLatitude().doubleValue(), bestPoint.getLongitude().doubleValue());
    }

    private Trip createTripBetweenVisits(User user, String previewId, ProcessedVisit startVisit, ProcessedVisit endVisit) {
        Instant tripStartTime = startVisit.getEndTime();
        Instant tripEndTime = endVisit.getStartTime();
        if (previewId != null) {
            if (this.previewProcessedVisitJdbcService.findById(startVisit.getId()).isEmpty() || this.previewProcessedVisitJdbcService.findById(endVisit.getId()).isEmpty()) {
                logger.debug("One of the following preview visits [{},{}] where already deleted. Will skip trip creation.", (Object)startVisit.getId(), (Object)endVisit.getId());
                return null;
            }
        } else if (this.processedVisitJdbcService.findById(startVisit.getId()).isEmpty() || this.processedVisitJdbcService.findById(endVisit.getId()).isEmpty()) {
            logger.debug("One of the following visits [{},{}] where already deleted. Will skip trip creation.", (Object)startVisit.getId(), (Object)endVisit.getId());
            return null;
        }
        if (tripEndTime.isBefore(tripStartTime) || tripEndTime.equals(tripStartTime)) {
            logger.warn("Invalid trip time range detected for user {}: {} to {}", new Object[]{user.getUsername(), tripStartTime, tripEndTime});
            return null;
        }
        if (previewId == null && this.tripJdbcService.existsByUserAndStartTimeAndEndTime(user, tripStartTime, tripEndTime)) {
            logger.debug("Trip already exists for user {} from {} to {}", new Object[]{user.getUsername(), tripStartTime, tripEndTime});
            return null;
        }
        List tripPoints = previewId == null ? this.rawLocationPointJdbcService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, tripStartTime, tripEndTime) : this.previewRawLocationPointJdbcService.findByUserAndTimestampBetweenOrderByTimestampAsc(user, previewId, tripStartTime, tripEndTime);
        double estimatedDistanceInMeters = this.calculateDistanceBetweenPlaces(startVisit.getPlace(), endVisit.getPlace());
        double travelledDistanceMeters = GeoUtils.calculateTripDistance((List)tripPoints);
        TransportMode transportMode = this.transportModeService.inferTransportMode(user, tripPoints, tripStartTime, tripEndTime);
        Trip trip = new Trip(tripStartTime, tripEndTime, Long.valueOf(tripEndTime.getEpochSecond() - tripStartTime.getEpochSecond()), Double.valueOf(estimatedDistanceInMeters), Double.valueOf(travelledDistanceMeters), transportMode, startVisit, endVisit);
        logger.debug("Created trip from {} to {}: travelled distance={}m, mode={}", new Object[]{Optional.ofNullable(startVisit.getPlace().getName()).orElse("Unknown Name"), Optional.ofNullable(endVisit.getPlace().getName()).orElse("Unknown Name"), Math.round(travelledDistanceMeters), transportMode});
        return trip;
    }

    private List<SignificantPlace> findNearbyPlaces(User user, String previewId, double latitude, double longitude, DetectionParameter.VisitMerging mergeConfiguration) {
        Point point = this.geometryFactory.createPoint(new Coordinate(longitude, latitude));
        if (previewId == null) {
            return this.significantPlaceJdbcService.findEnclosingPlaces(user.getId(), point, GeoUtils.metersToDegreesAtPosition((double)((double)mergeConfiguration.getMinDistanceBetweenVisits() / 2.0), (double)latitude));
        }
        return this.previewSignificantPlaceJdbcService.findNearbyPlaces(user.getId(), point, GeoUtils.metersToDegreesAtPosition((double)((double)mergeConfiguration.getMinDistanceBetweenVisits() / 2.0), (double)latitude), previewId);
    }

    private SignificantPlace findOrCreateSignificantPlace(User user, String previewId, double latitude, double longitude, DetectionParameter.VisitMerging mergeConfig, String traceId) {
        List nearbyPlaces = this.findNearbyPlaces(user, previewId, latitude, longitude, mergeConfig);
        return nearbyPlaces.isEmpty() ? this.createSignificantPlace(user, latitude, longitude, previewId, traceId) : this.findClosestPlace(latitude, longitude, nearbyPlaces);
    }

    private SignificantPlace createSignificantPlace(User user, double latitude, double longitude, String previewId, String traceId) {
        GeoPoint point;
        Optional override;
        SignificantPlace significantPlace = SignificantPlace.create((Double)latitude, (Double)longitude);
        Optional timezone = this.timezoneService.getTimezone(significantPlace);
        if (timezone.isPresent()) {
            significantPlace = significantPlace.withTimezone((ZoneId)timezone.get());
        }
        if ((override = this.significantPlaceOverrideJdbcService.findByUserAndPoint(user, point = new GeoPoint(significantPlace.getLatitudeCentroid().doubleValue(), significantPlace.getLongitudeCentroid().doubleValue()))).isPresent()) {
            logger.info("Found override for user [{}] and location [{}], using override information: {}", new Object[]{user.getUsername(), point, override.get()});
            significantPlace = significantPlace.withName(((PlaceInformationOverride)override.get()).name()).withType(((PlaceInformationOverride)override.get()).category()).withTimezone(((PlaceInformationOverride)override.get()).timezone()).withPolygon(((PlaceInformationOverride)override.get()).polygon());
        }
        significantPlace = previewId == null ? this.significantPlaceJdbcService.create(user, significantPlace) : this.previewSignificantPlaceJdbcService.create(user, previewId, significantPlace);
        this.publishSignificantPlaceCreatedEvent(user, significantPlace, previewId, traceId);
        return significantPlace;
    }

    private SignificantPlace findClosestPlace(double latitude, double longitude, List<SignificantPlace> places) {
        Comparator<SignificantPlace> distanceComparator = Comparator.comparingDouble(place -> GeoUtils.distanceInMeters((double)latitude, (double)longitude, (double)place.getLatitudeCentroid(), (double)place.getLongitudeCentroid()));
        return places.stream().min(distanceComparator.thenComparing(SignificantPlace::getId)).orElseThrow(() -> new IllegalStateException("No places found"));
    }

    private double calculateDistanceBetweenPlaces(SignificantPlace place1, SignificantPlace place2) {
        return GeoUtils.distanceInMeters((double)place1.getLatitudeCentroid(), (double)place1.getLongitudeCentroid(), (double)place2.getLatitudeCentroid(), (double)place2.getLongitudeCentroid());
    }

    private void publishSignificantPlaceCreatedEvent(User user, SignificantPlace place, String previewId, String traceId) {
        SignificantPlaceCreatedEvent event = new SignificantPlaceCreatedEvent(user.getUsername(), previewId, place.getId(), place.getLatitudeCentroid(), place.getLongitudeCentroid(), traceId);
        this.rabbitTemplate.convertAndSend("reitti-exchange", "reitti.place.created.v2", (Object)event);
        logger.info("Published SignificantPlaceCreatedEvent for place ID: {}", (Object)place.getId());
    }
}

